#!/usr/bin/env python
# -*- coding: utf-8 -*-
########################################################################
# version 1.2.0
#
# GimpPythonFuExtenedOpenRsater.py
# Copyright (C) 2015 かんら・から http://www.pixiv.net/member.php?id=3098715
# 
# GimpPythonFuLayerAction-2-8.py is Python-fu plugin for GIMP 2.8
#
# GimpPythonFuLayerAction-2-8.py is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#  
# GimpPythonFuLayerAction-2-8.py is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
# 
# GPLv3 ライセンス
# かんら・から http://www.pixiv.net/member.php?id=3098715
# バグレポート・アイデアなどは pixiv メッセージでお願いします。
#
# ダウンロード
# http://www.magic-object.mydns.jp/
#
# このスクリプトを使用して発生した問題に関し、作成者は如何なる保証などを行う事はありません。
# 自己責任でのみの使用を許可します。
########################################################################
from gimpfu import *
########################################################################
import re
import tempfile
import zipfile
import os
import xml
import xml.etree.ElementTree as XmlElementTree
########################################################################
# OpenRaster操作クラス
#
#
#
class OpenRaster:
	########################################################################
	stackImageElement = None
	imageVersion = ''
	imageWidth = 0
	imageHeight = 0
	xResorution = 72
	yResorution = 72
	elementDict = {}
	thumbnailBuf = None
	xOffset = 0
	yOffset = 0
	########################################################################
	def __init__( self ):
		return
	########################################################################
	def load( self, path=None ) :
		oraDir = None
		self.elementDict = {}
		self.thumbnailBuf = None
		self.xOffset = 0
		self.yOffset = 0
		try :
		########################################################################
			if not path :
				return False
			elif not os.path.isfile( path ) :
				return False
			elif os.path.getsize( path ) < 1 :
				return False
			elif not zipfile.is_zipfile( path ) :
				return False
			########################################################################
			zipOraFile = zipfile.ZipFile( path, 'r' )

			stackXmlString = zipOraFile.read( 'stack.xml' )
			if not stackXmlString :
				return False

			self.stackImageElement =  XmlElementTree.fromstring( stackXmlString )
			if self.stackImageElement.tag != 'image' :
				self.stackImageElement = None
				return False
			
			self.elementDict = self.itemParse( self.stackImageElement, zipOraFile )
			if len( self.elementDict ) < 1 :
				return False
			########################################################################
			try :
				self.thumbnailBuf = zipOraFile.read( 'Thumbnails/thumbnail.png' )
			except :
				self.thumbnailBuf = None
			########################################################################
			return True
			########################################################################
		########################################################################
		except :
			self.stackImageElement = None
			self.itemTree = []
			return False
		########################################################################
		return False
	########################################################################
	def itemParse( self, element, zipOraFile ) :
		########################################################################
		if not isinstance( element, xml.etree.ElementTree.Element ) :
			return {}
		elif not isinstance( zipOraFile, zipfile.ZipFile ) :
			return {}
		########################################################################
		try :
			########################################################################
			attributesDict = {}
			attributesDict['name'] = element.get( 'name', '' )
			attributesDict['opacity'] = float( element.get( 'opacity', 1 ) )
			attributesDict['visibility'] = True if element.get('visibility', 'visible') == 'visible' else False
			attributesDict['x'] = int( element.get( 'x', 0 ) )
			attributesDict['y'] = int( element.get( 'y', 0 ) )
			attributesDict['layerMode'] = element.get( 'composite-op', 'svg:src-over' )
			attributesDict['hasChild'] = False if len( list( element ) ) < 1 else True
			attributesDict['src'] = element.get( 'src', '' )
			attributesDict['childList'] = []

			attributesDict['imageBuf'] = None
			attributesDict['value'] = None
			attributesDict['imageType'] = None

			self.xOffset += attributesDict['x']
			self.yOffset += attributesDict['y']

			attributesDict['xOffset'] = self.xOffset
			attributesDict['yOffset'] = self.yOffset
			########################################################################
			if attributesDict['src'].lower().endswith('.png') :
				attributesDict['imageType'] = 'PNG'
			elif attributesDict['src'].lower().endswith('.svg') :
				attributesDict['imageType'] = 'SVG'
			elif attributesDict['src'].lower().endswith('.jpeg') or attributesDict['src'].lower().endswith('.jpg') :
				attributesDict['imageType'] = 'JPEG'
			elif attributesDict['src'].lower().endswith('.tiff') or attributesDict['src'].lower().endswith('.tif') :
				attributesDict['imageType'] = 'TIFF'
			elif len( attributesDict['src'] ) > 0 :
				attributesDict['imageType'] = 'Unkown'

			if len( attributesDict['src'] ) > 0 :
				attributesDict['imageBuf'] = zipOraFile.read( attributesDict['src'] )
			########################################################################			
			for key in element.keys() :
				if key in attributesDict.keys() :
					continue
			
				value = element.get( key, '' )
				try :
					attributesDict[ key ] = int( value )
				except :
					attributesDict[ key ] = value
			########################################################################			
			attributesDict['elementType'] = element.tag
			if element.tag == 'image' :
				self.imageVersion = element.get( 'version', 0 )
				self.imageWidth = int( element.get( 'w', 0 ) )
				self.imageHeight = int( element.get( 'h', 0 ) )
				self.xResorution = int( element.get( 'xres', 72 ) )
				self.yResorution = int( element.get( 'yres', 72 ) )
			elif element.tag == 'stack' :
				attributesDict['elementType'] = 'group'
			elif element.tag == 'text' :
				attributesDict['value'] = element.text
				return attributesDict
			########################################################################			
			if attributesDict['hasChild'] :
				for childElement in list( element ):
					attributesDict['childList'].append( self.itemParse( childElement, zipOraFile ) )
			########################################################################			
			return attributesDict
		########################################################################
		except :
			return {}
		########################################################################
		return {}
########################################################################
def ExtnedOpenRasterThumbnailer( path, thumb_size ):
	ora = OpenRaster()
	elementDict = ora.load( path )
	
	if ora.thumbnailBuf is None :
		return ( None, ora.imageWidth, ora.imageHeight )
	
	tempDirPath = tempfile.mkdtemp( 'ExtnedOpenRasterThumbnailer' )
	tempFilePath = os.path.join( tempDirPath, 'thumbnail.png' )

	with open( tempFilePath, 'wb' ) as fp :
		fp.write( ora.thumbnailBuf )

	image = pdb['file-png-load']( tempFilePath )

	os.remove( tempFilePath )
	os.rmdir( tempDirPath )

	return ( image, ora.imageWidth, ora.imageHeight )
########################################################################
class OpenRasterTool :
	########################################################################
	ora = None
	skipTopGroup = True
	image = None
	path = None
	layerModeMap = {
		'svg:src-over'	: NORMAL_MODE,	# 標準
		'svg:plus'		: ADDITION_MODE,	# 加算
		'svg:multiply'	: MULTIPLY_MODE,	# 乗算 
		'svg:screen'		: SCREEN_MODE,	# スクリーン
		'svg:overlay'		: OVERLAY_MODE,	# オーバーレイ
		'svg:darken'		: DARKEN_ONLY_MODE,	# 比較(暗)
		'svg:lighten'		: LIGHTEN_ONLY_MODE,	# 比較(明)
		'svg:color-dodge'	: DODGE_MODE,		# 覆い焼き
		'svg:color-burn'	: BURN_MODE,		# 焼き込み
		'svg:hard-light'	: HARDLIGHT_MODE,	# ハードライト
		'svg:soft-light'	: SOFTLIGHT_MODE,	# ソフトライト
		'svg:difference'	: DIFFERENCE_MODE,	# 差の絶対値
		'svg:color'		: COLOR_MODE,		# 色
		'svg:luminosity'	: VALUE_MODE,		# ≒ 明度
		'svg:hue'		: HUE_MODE,		# 色相
		'svg:saturation'	: SATURATION_MODE,	# 彩度
		'gimp:dissolve'	: DISSOLVE_MODE,	# ディザー合成
		'gimp:divide'		: DIVIDE_MODE,	# 除算
		'gimp:grain-extract'	: GRAIN_EXTRACT_MODE,# 微粒取り出し
		'gimp:grain-merge'	: GRAIN_MERGE_MODE,	# 微粒結合
		'gimp:subtract'	: GRAIN_MERGE_MODE,	# 減算
		'gimp:color-erase'	: 22, # COLOR-ERASE-MODE 動作するが…
		'gimp:erase'		: 23, # ERASE-MODE 動作するが…
		'gimp:replace'	: 24, # REPLACE-MODE 動作するが…
		'gimp:anti-erase'	: 25, # ANTI-ERASE-MODE 動作するが…
		'krita:erase'		: 23, # ERASE-MODE 動作するが…
	}
	layerCkipMode = {
		'svg:dst-in',		# Destination In
		'svg:dst-out',	# Destination Out
		'svg:src-atop',	# Source Atop
		'svg:dst-atop'	# Destination Atop
	}
	layerClipDepthMap = {};
	allowImageType = [ 'PNG', 'JPEG', 'TIFF', 'SVG' ]
	mask = None
	saveItemCounter = 0
	########################################################################
	doSaveLayerMask = False
	doSaveVectors = True
	doLoadSvgAsVectors = True
	doCleanupEmptyGroup = True
	doMoveParentSingleItemInGroup = True
	doMoveParentSourceAtopItemInGroup = True
	########################################################################
	def __init__( self ) :
		#print 'start create OpenRaster'
		self.ora = OpenRaster()
		#print 'start created OpenRaster'
		return
	########################################################################
	def __del__( self ) :
		if self.image is not None :
			pdb['gimp-image-undo-enable']( self.image )
	########################################################################
	def load( self, path ) :
		#print 'start create OpenRaster.load( path ):' + path
		if not self.ora.load( path ) :
			return None

		#print 'end create OpenRaster.load( path ):' + path
		self.path = path
		self.skipTopGroup = True
		self.mask = None
		self.layerClipDepthMap = {}

		#print 'start create elementToLayer'
		self.elementToLayer( self.ora.elementDict )
		#print 'end create elementToLayer'

		self.setClipMask()

		if self.doMoveParentSingleItemInGroup :
			self.moveParentSingleItemInGroup()

		if self.doCleanupEmptyGroup :
			self.cleanupEmptyGroup()

		return self.image
	########################################################################
	def save( self, image, drawable, path ) :
		if image is None :
			return False
		elif path is None :
			return False
		elif not isinstance( path, str ) :
			return False
		elif len( path ) < 1 :
			return False
		
		( count, idList ) = pdb['gimp-image-get-layers']( image )
		if count < 1 :
			return False

		imageElement = XmlElementTree.Element( 'image' )
		imageElement.set( 'version', '0.0.1' )
		imageElement.set( 'w', str( pdb['gimp-image-width']( image ) ) )
		imageElement.set( 'h', str( pdb['gimp-image-height']( image ) ) )

		( xResolution, yResolution ) = pdb['gimp-image-get-resolution']( image )
		imageElement.set( 'xres', str( int( xResolution ) ) )
		imageElement.set( 'yres', str( int( yResolution ) ) )

		topStackElement = XmlElementTree.SubElement( imageElement, 'stack' )

		zipOraFile = zipfile.ZipFile( path, 'w', compression=zipfile.ZIP_STORED )

		( count, layerIdList ) = pdb['gimp-image-get-layers']( image )
		for layerId in layerIdList :
			layer = gimp.Item.from_id( layerId )
			if not self.storeLayers( image, layer, topStackElement, zipOraFile ) :
				zipOraFile.cloae()
				return False

		if not self.saveString( 'image/openraster', 'mimetype', zipOraFile ) :
			zipOraFile.cloae()
			return False

		#print 'Start: saveVector'
		if not self.saveVector( image, topStackElement, zipOraFile ) :
			zipOraFile.cloae()
			return False
		#print 'end: saveVector'

		xmlString = XmlElementTree.tostring( imageElement, encoding='UTF-8' )
		if not self.saveString( xmlString, 'stack.xml', zipOraFile ) :
			zipOraFile.cloae()
			return False

		if not self.saveThumbnail( image, zipOraFile ) :
			zipOraFile.cloae()
			return False

		zipOraFile.close()

		return True
	########################################################################
	def getLayerDepth( self, layer ) :
		if self.image is None :
			return -1
		elif layer is None :
			return -1

		isLayer = False

		if pdb['gimp-item-is-group']( layer ) :
			isLayer = True
		elif pdb['gimp-item-is-layer']( layer ) :
			isLayer = True
		elif pdb['gimp-item-is-text-layer']( layer ) :
			isLayer = True
		else :
			return -1

		depth = 0
		parent = pdb['gimp-item-get-parent']( layer )

		while parent is not None :
			depth += 1
			parent = pdb['gimp-item-get-parent']( parent )

		return depth
	########################################################################
	def setClipDepthMap( self, layer, layerMode ) :
		if self.image is None :
			return False
		elif layer is None :
			return False
		elif not isinstance( layerMode, str ) :
			return False
		elif layerMode not in self.layerCkipMode :
			return False

		depth = self.getLayerDepth( layer )
		if depth < 0 :
			return False

		if not isinstance( self.layerClipDepthMap, dict ) :
			self.layerClipDepthMap = {}

		if depth not in self.layerClipDepthMap :
			self.layerClipDepthMap[ depth ] = []

		self.layerClipDepthMap[ depth ].insert( 0, { 'layer' : layer, 'layerMode' : layerMode } )

		return True
	########################################################################
	def clipFromSelection( self, layer ) :
		if self.image is None :
			return False
		if pdb['gimp-selection-is-empty']( self.image ) :
			#print "empty selection"
			return False
		
		if pdb['gimp-item-is-group']( layer ) :
			#print "target is group : " + pdb['gimp-item-get-name']( layer )
			( count, idList ) = pdb['gimp-item-get-children']( layer )
			for id in idList :
				if not self.clipFromSelection( gimp.Item.from_id( id ) ) :
					return False
			return True
		elif pdb['gimp-item-is-layer']( layer ) or pdb['gimp-item-is-text-layer']( layer ) :

			#print "target is layer : " + pdb['gimp-item-get-name']( layer )

			if isinstance( pdb['gimp-layer-get-mask']( layer ), gimp.Channel ) :
				# レイヤーマスクが存在する
				# レイヤーマスクを反映
				pdb['gimp-layer-remove-mask']( layer, MASK_APPLY )

			# 選択範囲をマスクに設定
			mask = pdb['gimp-layer-create-mask']( layer, ADD_SELECTION_MASK )
			pdb['gimp-layer-add-mask']( layer, mask )

			return True
		else :
			return False
	########################################################################
	def selectSourceAtop( self, layer ) :
		if self.image is None :
			return False

		if not pdb['gimp-item-is-group']( layer ) :
			if not pdb['gimp-item-is-layer']( layer ) :
				if not pdb['gimp-item-is-text-layer']( layer ) :
					return False
		
		parent = pdb['gimp-item-get-parent']( layer )

		count = 1
		if parent is None :
			( count, idList ) = pdb['gimp-image-get-layers']( self.image )
		else :
			( count, idList ) = pdb['gimp-item-get-children']( parent )

		if count < 2 :
			return False

		position = pdb['gimp-image-get-item-position']( self.image, layer )
		
		if ( position + 1 ) >= count :
			# position がグループの末尾である場合、意味がないので無視
			# MyPaint には pass-thrugh 機能があるが、GIMP には存在しない。
			return False

		pdb['gimp-selection-none']( self.image )
		
	########################################################################
	def setClipMask( self ) :
		if self.image is None :
			return False
		if not isinstance( self.layerClipDepthMap, dict ) :
			return False

		depthKeysList = self.layerClipDepthMap.keys()
		if len( depthKeysList ) < 1 :
			return True

		depthKeysList.sort()

		depthKeysList.reverse()

		#print "depthKeysList:" + str( depthKeysList )

		# 深い順でクリップマスクを作成
		for depth in depthKeysList :
			for sameDepthList in self.layerClipDepthMap[ depth ] :
				########################################################################
				parent = pdb['gimp-item-get-parent']( sameDepthList['layer'] )

				count = 1
				if parent is None :
					( count, idList ) = pdb['gimp-image-get-layers']( self.image )
				else :
					( count, idList ) = pdb['gimp-item-get-children']( parent )

				if count < 1 :
					continue
				########################################################################
				position = pdb['gimp-image-get-item-position']( self.image, sameDepthList['layer'] )

				if ( position + 1 ) >= count :
					# position がグループの末尾である場合、意味がないので無視
					# MyPaint には pass-thrugh 機能があるが、GIMP には存在しない。
					continue

				targetLayerIdList = list( idList )

				targetLayerIdList = targetLayerIdList[ position + 1 : ]

				if sameDepthList['layerMode'] == 'svg:dst-in' :
					########################################################################
					if count >= 2 :
						# アルファマスクを選択
						pdb['gimp-image-select-item']( self.image, CHANNEL_OP_REPLACE, sameDepthList['layer'] )

						# 'svg:dst-in' ではレイヤー自体が不要なので削除
						pdb['gimp-image-remove-layer']( self.image, sameDepthList['layer'] )
					else :
						# 'svg:dst-in' ではレイヤー自体が不要なので削除
						pdb['gimp-image-remove-layer']( self.image, sameDepthList['layer'] )
						continue
					########################################################################
				elif sameDepthList['layerMode'] == 'svg:dst-out' :
					########################################################################
					if count >= 2 :
						# アルファマスクを選択
						pdb['gimp-image-select-item']( self.image, CHANNEL_OP_REPLACE, sameDepthList['layer'] )
						# 選択範囲を反転
						pdb['gimp-selection-invert']( self.image )

						# 'svg:dst-out' ではレイヤー自体が不要なので削除
						pdb['gimp-image-remove-layer']( self.image, sameDepthList['layer'] )
					else :
						# 'svg:dst-out' ではレイヤー自体が不要なので削除
						pdb['gimp-image-remove-layer']( self.image, sameDepthList['layer'] )
						continue
					########################################################################
				elif sameDepthList['layerMode'] == 'svg:src-atop' :
					########################################################################
					# アルファマスクを選択
					pdb['gimp-selection-none']( self.image )
					targetLayerIdList.reverse()
					# 末尾から選択範囲を合成していく
					for id in targetLayerIdList :
						layer = gimp.Item.from_id( id )
						pdb['gimp-image-select-item']( self.image, CHANNEL_OP_ADD, layer )

					self.clipFromSelection( sameDepthList['layer'] )
					pdb['gimp-selection-none']( self.image )

					if self.doMoveParentSourceAtopItemInGroup :
						#print "call >> raiseLayersToParent : " + pdb['gimp-item-get-name']( sameDepthList['layer'] )
						self.raiseLayersToParent( sameDepthList['layer'] )

					continue
					########################################################################
				elif sameDepthList['layerMode'] == 'svg:dst-atop' :
					########################################################################
					# アルファマスクを選択
					pdb['gimp-image-select-item']( self.image, CHANNEL_OP_REPLACE, sameDepthList['layer'] )
					########################################################################
					targetLayerIdList.reverse()

					# 末尾からレイヤーマスクを設定していく
					for id in targetLayerIdList :
						layer = gimp.Item.from_id( id )
						self.clipFromSelection( layer )

					pdb['gimp-selection-none']( self.image )
					########################################################################
					# アルファマスクを選択
					pdb['gimp-image-select-item']( self.image, CHANNEL_OP_REPLACE, sameDepthList['layer'] )

					# Destination の表示部分を選択範囲から引く
					for id in targetLayerIdList :
						layer = gimp.Item.from_id( id )
						pdb['gimp-image-select-item']( self.image, CHANNEL_OP_SUBTRACT, layer )

					self.clipFromSelection( sameDepthList['layer'] )

					pdb['gimp-selection-none']( self.image )
					
					continue
				########################################################################
				targetLayerIdList.reverse()

				# 末尾からレイヤーマスクを設定していく
				for id in targetLayerIdList :
					layer = gimp.Item.from_id( id )
					self.clipFromSelection( layer )

				pdb['gimp-selection-none']( self.image )
				########################################################################
				
		return True
	########################################################################
	def setLayerName( self, layer, name ) :
		if self.image is None :
			return None
		elif layer is None :
			return None
		
		if str is None or len( name ) < 1 :
			if pdb['gimp-item-is-group']( layer ) :
				name = 'Group'
			elif pdb['gimp-item-is-text-layer']( layer ) :
				name = 'text'
			else :
				name = 'Layer'

		if not pdb['gimp-image-get-layer-by-name']( self.image, name ) :
			pdb['gimp-item-set-name']( layer, name )
			return pdb['gimp-item-get-name']( layer )
		
		index = 0
		while pdb['gimp-image-get-layer-by-name']( self.image, name + str( index ) ) :
			index += 1
		
		pdb['gimp-item-set-name']( layer, name + str( index ) )

		return pdb['gimp-item-get-name']( layer )
	########################################################################
	def elementToLayer( self, elementDict, parent = None ) :
		########################################################################
		if elementDict['elementType'] == 'layer' :
			########################################################################
			if self.image is None :
				return
			elif elementDict['imageBuf'] is None :
				self.mask = None
				return
			elif elementDict['imageType'] == 'SVG' and self.doLoadSvgAsVectors :
				if not isinstance( elementDict['imageBuf'], str ) :
					self.mask = None
					return
				( count, vectorIdList ) = pdb['gimp-vectors-import-from-string']( self.image, elementDict['imageBuf'], len(elementDict['imageBuf']), True, False )
				if count > 0 :
					vecotr = gimp.Item.from_id( vectorIdList[0] )
					pdb['gimp-item-set-name']( vecotr, elementDict['name'] )
					pdb['gimp-item-set-visible']( vecotr, elementDict['visibility'] )
				
				self.mask = None
				return
			elif elementDict['imageBuf'] is not None :
				layer = self.buffToLayer( elementDict['imageBuf'], elementDict['imageType'] )
				if layer is None :
					return

				self.setLayerName( layer, elementDict['name'] )
				pdb['gimp-layer-set-offsets']( layer, elementDict['x'], elementDict['y'] )
				pdb['gimp-layer-set-opacity']( layer, elementDict['opacity'] * 100 )
				pdb['gimp-item-set-visible']( layer, elementDict['visibility'] )
				if elementDict['layerMode'] in self.layerModeMap.keys() :
					pdb['gimp-layer-set-mode']( layer, self.layerModeMap[ elementDict['layerMode'] ] )
				
				count = 1
				if parent is None :
					( count, idList ) = pdb['gimp-image-get-layers']( self.image )
				else :
					( count, idList ) = pdb['gimp-item-get-children']( parent )
			
				pdb['gimp-image-insert-layer']( self.image, layer, parent, count )
				#print 'Insert Layer ' + str( elementDict['name'] ) + ' > ' + str(count) + " visible:" + str(elementDict['visibility'])

				self.setClipDepthMap( layer, elementDict['layerMode'] )

				return
			else :
				self.mask = None
				return
			########################################################################
		elif elementDict['elementType'] == 'text' :
			if elementDict['value'] is None :
				return

			fontName = pdb['gimp-context-get-font']()
			fontSize = 32
			fontUnit = 0

			for key in elementDict.keys() :
				########################################################################
				if re.search( 'font[-_]?name$', key.lower(), re.I ) :
					fontName = elementDict[ key ]
				########################################################################
				if re.search( 'font[-_]?size$', key.lower(), re.I ) :
					sizeStrRsults = re.match( '^(\d+)\s*([A-Za-z]+)', elementDict[ key ] )
					if not sizeStrRsults :
						continue
					fontSize = int( sizeStrRsults.group(1) )
					unitName = sizeStrRsults.group(2).lower()
					for unitId in range( pdb['gimp-unit-get-number-of-units']() ) :
						if unitName == pdb['gimp-unit-get-abbreviation']( unitId ).lower() :
							fontUnit = unitId
							break
				########################################################################

			textLayer = pdb['gimp-text-layer-new']( self.image, elementDict['value'], fontName, fontSize, fontUnit )
			self.setLayerName( layer, elementDict['name'] )
			pdb['gimp-layer-set-offsets']( textLayer, elementDict['x'], elementDict['y'] )
			pdb['gimp-layer-set-opacity']( textLayer, elementDict['opacity'] * 100 )
			pdb['gimp-item-set-visible']( textLayer, elementDict['visibility'] )
			if elementDict['layerMode'] in self.layerModeMap.keys() :
				pdb['gimp-layer-set-mode']( textLayer, self.layerModeMap[ elementDict['layerMode'] ] )

			pdb['gimp-text-layer-set-text']( textLayer, elementDict['value'] )

			count = 1
			if parent is None :
				( count, idList ) = pdb['gimp-image-get-layers']( self.image )
			else :
				( count, idList ) = pdb['gimp-item-get-children']( parent )
		
			pdb['gimp-image-insert-layer']( self.image, textLayer, parent, count )

			self.setClipDepthMap( textLayer, elementDict['layerMode'] )

			return
			########################################################################
		else :
			########################################################################
			if elementDict['elementType'] == 'image' :
				self.image = pdb['gimp-image-new']( self.ora.imageWidth, self.ora.imageHeight, RGB )
				pdb['gimp-image-undo-disable']( self.image )
				pdb['gimp-image-set-filename']( self.image, self.path )
				pdb['gimp-image-set-resolution']( self.image, self.ora.xResorution, self.ora.yResorution )
				parent = None
			elif elementDict['elementType'] == 'group' :
				if self.image is None :
					return
				if self.skipTopGroup and elementDict['x'] == 0 and elementDict['y'] == 0 :
					self.skipTopGroup = False
				else :
					self.skipTopGroup = False

					layerGroup = pdb['gimp-layer-group-new']( self.image )
					self.setLayerName( layerGroup, elementDict['name'] )
					pdb['gimp-layer-set-offsets']( layerGroup, elementDict['x'], elementDict['y'] )
					pdb['gimp-layer-set-opacity']( layerGroup, elementDict['opacity'] * 100 )
					pdb['gimp-item-set-visible']( layerGroup, elementDict['visibility'] )
					if elementDict['layerMode'] in self.layerModeMap.keys() :
						pdb['gimp-layer-set-mode']( layerGroup, self.layerModeMap[ elementDict['layerMode'] ] )

					count = 1
					if parent is None :
						( count, idList ) = pdb['gimp-image-get-layers']( self.image )
					else :
						( count, idList ) = pdb['gimp-item-get-children']( parent )
				
					pdb['gimp-image-insert-layer']( self.image, layerGroup, parent, count )

					self.setClipDepthMap( layerGroup, elementDict['layerMode'] )

					parent = layerGroup
			########################################################################
			if elementDict['hasChild'] :
				for child in elementDict['childList'] :
					self.elementToLayer( child, parent )
			########################################################################
	########################################################################
	def addLayerMask( self, layer ) :
		if self.image is None :
			return None
		elif layer is None :
			return
		elif self.mask is None :
			return
		elif self.mask['imageType'] not in self.allowImageType :
			self.mask = None
			return None
		elif self.mask['imageBuf'] is None :
			return layer

		imageType = self.mask['imageType']
		
		tempDirPath = tempfile.mkdtemp( 'ExtnedOpenRasterLayerMask' )
		tempFilePath = os.path.join( tempDirPath, 'mask.' + imageType.lower() )

		with open( tempFilePath, 'wb' ) as fp :
			fp.write( self.mask['imageBuf'] )

		maskItem = None

		maskLayer = pdb['gimp-file-load-layer']( self.image, tempFilePath )
		layerMask = pdb['gimp-layer-create-mask']( maskLayer, ADD_COPY_MASK )

		if self.mask['layerMode'] == 'svg:dst-in' :
			pdb['gimp-layer-add-mask']( layer, layerMask )
			pdb['gimp-item-delete']( maskLayer )
		elif self.mask['layerMode'] == 'svg:dst-out' :
			orgAlphLayerMask = pdb['gimp-layer-create-mask']( layer, ADD_ALPHA_MASK )
			pdb['gimp-channel-combine-masks']( layerMask, orgAlphLayerMask, CHANNEL_OP_SUBTRACT, 0, 0 )
			pdb['gimp-layer-remove-mask']( maskLayer, MASK_DISCARD )
			pdb['gimp-layer-add-mask']( maskLayer, layerMask )
			pdb['gimp-item-delete']( layer )
			layer = maskLayer

		os.remove( tempFilePath )
		os.rmdir( tempDirPath )

		self.mask = None

		return layer
	########################################################################
	def buffToLayer( self, buff, imageType ) :
		if self.image is None :
			return None
		if buff is None :
			return None
		elif not isinstance( imageType, str ) :
			return None
		elif imageType not in self.allowImageType :
			return None

		tempDirPath = tempfile.mkdtemp( 'ExtnedOpenRasterLayer' )
		tempFilePath = os.path.join( tempDirPath, 'layer.' + imageType.lower() )

		with open( tempFilePath, 'wb' ) as fp :
			fp.write( buff )

		layer = pdb['gimp-file-load-layer']( self.image, tempFilePath )

		os.remove( tempFilePath )
		os.rmdir( tempDirPath )

		#if self.mask is not None :
		#	layer = self.addLayerMask( layer )

		return layer
	########################################################################
	def cleanupEmptyGroup( self ) :
		if not isinstance( self.image, gimp.Image ) :
			return False

		( count, layerIdList ) = pdb['gimp-image-get-layers']( self.image )
		if count < 1 :
			return True

		for listId in layerIdList :
			layer = gimp.Item.from_id( listId )
			if not pdb['gimp-item-is-group']( layer ) :
				continue
			self.cleanupEmptyGroupInGroup( layer )

		return True
	########################################################################
	def cleanupEmptyGroupInGroup( self, group ) :
		if not isinstance( self.image, gimp.Image ) :
			return False
		if not isinstance( self.image, gimp.Image ) :
			return False
		if group is None :
			return False

		if not pdb['gimp-item-is-group']( group ) :
			return True

		( count, layerIdList ) = pdb['gimp-item-get-children']( group )
		if count < 1 :
			pdb['gimp-image-remove-layer']( self.image, group )
			return True

		for listId in layerIdList :
			layer = gimp.Item.from_id( listId )
			if not pdb['gimp-item-is-group']( layer ) :
				continue
			
			if not self.cleanupEmptyGroupInGroup( layer ) :
				return False

		return True
	########################################################################
	def moveParentSingleItemInGroup( self ) :
		if not isinstance( self.image, gimp.Image ) :
			return False

		( count, layerIdList ) = pdb['gimp-image-get-layers']( self.image )
		if count < 1 :
			return True
		
		for listId in layerIdList :
			layer = gimp.Item.from_id( listId )
			if not pdb['gimp-item-is-group']( layer ) :
				continue

			self.moveParentSingleItemInGroupOfGroup( layer )

		return True
	########################################################################
	def moveParentSingleItemInGroupOfGroup( self, group ) :
		if not isinstance( self.image, gimp.Image ) :
			return False
		if group is None :
			return False

		if not pdb['gimp-item-is-group']( group ) :
			return True

		( count, layerIdList ) = pdb['gimp-item-get-children']( group )
		if count < 1 :
			return True
		if count == 1 :
			layer = gimp.Item.from_id( layerIdList[0] )
			if pdb['gimp-item-is-group']( layer ) :
				return self.moveParentSingleItemInGroupOfGroup( layer )
			else :
				parent = pdb['gimp-item-get-parent']( group )
				position = pdb['gimp-image-get-item-position']( self.image, group )
				pdb['gimp-image-reorder-item']( self.image, layer, parent, position )

				pdb['gimp-image-remove-layer']( self.image, group )
				return True

		for listId in layerIdList :
			layer = gimp.Item.from_id( listId )
			if not pdb['gimp-item-is-group']( layer ) :
				continue
			
			if not self.moveParentSingleItemInGroupOfGroup( layer ) :
				return False

		return True
	########################################################################
	def raiseLayersToParent( self, group ) :

		if not isinstance( self.image, gimp.Image ) :
			return False
		if group is None :
			return False
		if not pdb['gimp-item-is-group']( group ) :
			return True

		#print "raiseLayersToParent : " + pdb['gimp-item-get-name']( group )

		( count, layerIdList ) = pdb['gimp-item-get-children']( group )
		if count < 1 :
			if self.doCleanupEmptyGroup :
				pdb['gimp-image-remove-layer']( self.image, group )
				
			return True

		for listId in layerIdList :
			layer = gimp.Item.from_id( listId )

			parent = pdb['gimp-item-get-parent']( group )
			position = pdb['gimp-image-get-item-position']( self.image, group )
			pdb['gimp-image-reorder-item']( self.image, layer, parent, position )

		if self.doCleanupEmptyGroup :
			pdb['gimp-image-remove-layer']( self.image, group )

		return True
	########################################################################
	def storeLayers( self, image, layer, parentElement, zipOraFile ) :
		if image is None :
			return False
		if layer is None :
			return False
		elif not isinstance( parentElement, XmlElementTree.Element ) :
			return False
		elif not isinstance( zipOraFile, zipfile.ZipFile ) :
			return False

		opecity = pdb['gimp-layer-get-opacity']( layer ) / 100.0
		visibility = 'visible' if pdb['gimp-item-get-visible']( layer ) else 'hidden'
		name = pdb['gimp-item-get-name']( layer )
		( xOffset, yOffset ) = pdb['gimp-drawable-offsets']( layer )

		compositeOperation = 'svg:src-over'
		gimpLayerMode = pdb['gimp-layer-get-mode']( layer )

		for ( key, value ) in self.layerModeMap.items() :
			if value == gimpLayerMode :
				compositeOperation = key
				break
		
		if pdb['gimp-item-is-group']( layer ) :
			########################################################################
			stackElement = XmlElementTree.SubElement( parentElement, 'stack' )
			stackElement.set( 'opacity', str( opecity ) )
			stackElement.set( 'visibility', visibility )
			stackElement.set( 'name', name )
			stackElement.set( 'composite-op', compositeOperation )
			stackElement.set( 'x', str( xOffset ) )
			stackElement.set( 'y', str( yOffset ) )

			( count, childIdList ) = pdb['gimp-item-get-children']( layer )
			for childId in childIdList :
				childLayer = gimp.Item.from_id( childId )
				if not self.storeLayers( image, childLayer, stackElement, zipOraFile ) :
					return False
			return True
			########################################################################
		elif pdb['gimp-item-is-layer']( layer ) and self.doSaveLayerMask and isinstance( pdb['gimp-layer-get-mask']( layer ), gimp.Channel ) :
			########################################################################
			self.saveItemCounter += 1

			stackElement = XmlElementTree.SubElement( parentElement, 'stack' )
			
			layerMaskElement = XmlElementTree.SubElement( stackElement, 'layer' )
			layerMaskElement.set( 'name', name + '-mask' )
			layerMaskElement.set( 'composite-op', 'svg:dst-in' )
			# PNG ファイル自体が Offset を持つ為、「0」にする。 
			layerMaskElement.set( 'x', '0' )
			layerMaskElement.set( 'y', '0' )
			#layerMaskElement.set( 'x', str( xOffset ) )
			#layerMaskElement.set( 'y', str( yOffset ) )
			layerMaskElement.set( 'src', 'data/layer-' + str(self.saveItemCounter) + '-mask' + '.png' )

			if not self.saveLayerMask( image, layer, layerMaskElement, zipOraFile ) :
				return False

			layerElement = XmlElementTree.SubElement( stackElement, 'layer' )
			layerElement.set( 'opacity', str( opecity ) )
			layerElement.set( 'visibility', visibility )
			layerElement.set( 'name', name )
			layerElement.set( 'composite-op', compositeOperation )
			# PNG ファイル自体が Offset を持つ為、「0」にする。 
			layerElement.set( 'x', '0' )
			layerElement.set( 'y', '0' )
			#layerElement.set( 'x', str( xOffset ) )
			#layerElement.set( 'y', str( yOffset ) )
			layerElement.set( 'src', 'data/layer-' + str(self.saveItemCounter) + '.png' )
			
			return self.saveLayer( image, layer, layerElement, zipOraFile )
			########################################################################
		elif pdb['gimp-item-is-layer']( layer ) :
			########################################################################
			self.saveItemCounter += 1
			layerElement = XmlElementTree.SubElement( parentElement, 'layer' )
			layerElement.set( 'opacity', str( opecity ) )
			layerElement.set( 'visibility', visibility )
			layerElement.set( 'name', name )
			layerElement.set( 'composite-op', compositeOperation )
			# PNG ファイル自体が Offset を持つ為、「0」にする。 
			layerElement.set( 'x', '0' )
			layerElement.set( 'y', '0' )
			#layerElement.set( 'x', str( xOffset ) )
			#layerElement.set( 'y', str( yOffset ) )
			layerElement.set( 'src', 'data/layer-' + str(self.saveItemCounter) + '.png' )

			return self.saveLayer( image, layer, layerElement, zipOraFile )
			########################################################################
		elif pdb['gimp-item-is-text-layer']( layer ) :
			########################################################################
			textElement = XmlElementTree.SubElement( parentElement, 'layer' )
			textElement.set( 'opacity', str( opecity ) )
			textElement.set( 'visibility', visibility )
			textElement.set( 'name', name )
			textElement.set( 'composite-op', compositeOperation )
			textElement.set( 'x', str( xOffset ) )
			textElement.set( 'y', str( yOffset ) )

			textElement.text = pdb['gimp-text-layer-get-text']( layer )

			return True
			########################################################################
		else :
			return False
	########################################################################
	def saveLayerMask( self, image, layer, layerMaskElement, zipOraFile ) :
		if image is None :
			return False
		if layer is None :
			return False
		#elif not isinstance( layer, gimp.Layer ) :
		#	return False
		elif not isinstance( layerMaskElement, XmlElementTree.Element ) :
			return False
		elif 'src' not in layerMaskElement.keys() :
			return False
		elif not isinstance( zipOraFile, zipfile.ZipFile ) :
			return False
		elif not self.doSaveLayerMask :
			return True
		
		layerMask = pdb['gimp-layer-get-mask']( layer )
		if not isinstance(  layerMask, gimp.Channel ) :
			return True

		#maskedLayer = pdb['gimp-layer-copy']( layer, True )

		# fill the layer with BLACK
		#pdb['gimp-drawable-fill']( maskedLayer, WHITE_FILL )
		#pdb['gimp-image-add-layer-mask']( image, maskedLayer, layerMask )
		#pdb['gimp-layer-remove-mask']( maskedLayer, MASK_APPLY )

		tempDirPath = tempfile.mkdtemp( 'saveExtnedOpenRasterLayer' )
		tempFilePath = os.path.join( tempDirPath, 'layerMask.png' )
		
		#print 'saveLayerMask save to png file : ' + tempFilePath

		#pdb['file-png-save']( image, maskedLayer, tempFilePath, 'layerMask.png', False, 5, True, True, False, True, True )
		pdb['file-png-save']( image, layerMask, tempFilePath, 'layerMask.png', False, 5, True, True, False, True, True )
		zipOraFile.write( tempFilePath, layerMaskElement.get('src') )
		
		#pdb['gimp-item-delete']( maskedLayer )

		#print 'end saveLayerMask saveD to png file : ' + tempFilePath

		os.remove( tempFilePath )
		os.rmdir( tempDirPath )

		return True
	########################################################################
	def saveLayer( self, image, layer, layerElement, zipOraFile ) :
		if image is None :
			return False
		if layer is None :
			return False
		#elif not isinstance( layer, gimp.Layer ) :
		#	return False
		elif not isinstance( layerElement, XmlElementTree.Element ) :
			return False
		elif 'src' not in layerElement.keys() :
			return False
		elif not isinstance( zipOraFile, zipfile.ZipFile ) :
			return False
		
		tempDirPath = tempfile.mkdtemp( 'saveExtnedOpenRasterLayer' )
		tempFilePath = os.path.join( tempDirPath, 'layer.png' )
		
		#print 'saveLayer save to png file : ' + tempFilePath

		if not isinstance( pdb['gimp-layer-get-mask']( layer ), gimp.Channel ) : # No Layer-Mask
			pdb['file-png-save']( image, layer, tempFilePath, 'layer.png', False, 5, True, True, False, True, True )
		elif not self.doSaveLayerMask : # have Layer-Mask and "don't save Layer-Mask"
			copyedLayer = pdb['gimp-layer-copy']( layer, True )
			#cpoyedMask = pdb['gimp-channel-copy']( pdb['gimp-layer-get-mask']( layer ) )
			#pdb['gimp-layer-add-mask']( copyedLayer, cpoyedMask )
			pdb['gimp-image-insert-layer']( image, copyedLayer, None, 0 )
			pdb['gimp-layer-remove-mask']( copyedLayer, MASK_APPLY )
			pdb['file-png-save']( image, copyedLayer, tempFilePath, 'layer.png', False, 5, True, True, False, True, True )
			pdb['gimp-image-remove-layer']( image, copyedLayer )
		else : # have Layer-Mask and "save Layer-Mask"
			pdb['file-png-save']( image, layer, tempFilePath, 'layer.png', False, 5, True, True, False, True, True )
		
		zipOraFile.write( tempFilePath, layerElement.get('src') )
		
		#print 'end saveLayer saveD to png file : ' + tempFilePath

		os.remove( tempFilePath )
		os.rmdir( tempDirPath )

		return True
	########################################################################
	def saveString( self, buf, storePath, zipOraFile ) :
		if buf is None :
			return False
		elif not isinstance( storePath, str ) :
			return None
		elif len( storePath ) < 1 :
			return False
		elif not isinstance( zipOraFile, zipfile.ZipFile ) :
			return False

		zipOraFile.writestr( storePath, buf )
		
		return True
	########################################################################
	def saveVector( self, image, parentElement, zipOraFile ) :
		if image is None :
			return False
		elif not isinstance( parentElement, XmlElementTree.Element ) :
			return False
		elif not isinstance( zipOraFile, zipfile.ZipFile ) :
			return False

		( count, idList ) = pdb['gimp-image-get-vectors']( image )
		if count < 1 :
			return True

		vectorGroupElement = XmlElementTree.SubElement( parentElement, 'stack' )
		vectorGroupElement.set( 'visibility', 'hidden' )

		########################################################################
		for vectorId in idList :
			vector = gimp.Item.from_id( vectorId )
			vectorStr = pdb['gimp-vectors-export-to-string']( image, vector )
			storePath = 'data/vector-' + str(vectorId) + '.svg'

			#print 'save Vector Image ' + storePath

			if not self.saveString( vectorStr, storePath, zipOraFile ) :
				return False

			vectorElement = XmlElementTree.SubElement( vectorGroupElement, 'layer' )
			vectorElement.set( 'src', storePath )
			vectorElement.set( 'name', pdb['gimp-item-get-name']( vector ) )
			vectorElement.set( 'visibility', 'invisivle' )
			#print 'Add Vector Layer ' + storePath
		########################################################################
		return True
	########################################################################
	def saveThumbnail( self, image, zipOraFile ) :
		if image is None :
			return False
		elif not isinstance( zipOraFile, zipfile.ZipFile ) :
			return False
		
		imageWidth = pdb['gimp-image-width']( image )
		imageHeight = pdb['gimp-image-height']( image )

		scaleRate = 1.0 # scale rate for expand or shrink
		if imageWidth > imageHeight :
			scaleRate = 256.0 / float( imageWidth )
		else :
			scaleRate = 256.0 / float( imageWidth )

		thumbnailImage = pdb['gimp-image-duplicate']( image )
		thumnailLayer = pdb['gimp-image-flatten']( thumbnailImage )
		pdb['gimp-layer-scale']( thumnailLayer, int( imageWidth *  scaleRate ), int( imageHeight * scaleRate ), True )
		
		tempDirPath = tempfile.mkdtemp( 'saveExtnedOpenRasterThumbnail' )
		tempFilePath = os.path.join( tempDirPath, 'thumbnail.png' )
		
		#print 'saveThumbnail save to png file : ' + tempFilePath

		pdb['file-png-save']( thumbnailImage, thumnailLayer, tempFilePath, 'thumbnail.png', False, 5, True, True, False, True, True )
		#print 'end saveThumbnail save to png file : ' + tempFilePath

		zipOraFile.write( tempFilePath, 'Thumbnails/thumbnail.png' )
		
		pdb['gimp-image-delete']( thumbnailImage )

		os.remove( tempFilePath )
		os.rmdir( tempDirPath )

		return True
	########################################################################
########################################################################
# レイヤークリップクラス
#
#
#
class LayserClip:
	########################################################################
	image = None
	doPutMessage = True
	lastMessage = ''
	usableVersion = ( 2, 8, 0 )
	canUseLayerGroup = False
	layuersCount = 0
	processCount = 0
	########################################################################
	def __init__( self, image ):
		self.image = image
		self.layuersCount = 0

		versionLength = len( self.usableVersion )
		for index in range( 0, versionLength ) :
			if self.usableVersion[index] < pdb['gimp-version']()[index] :
				self.canUseLayerGroup = True
				break
			elif self.usableVersion[index] == pdb['gimp-version']()[index] :
				if index == ( versionLength - 1 ) :
					self.canUseLayerGroup = True
					break
				continue
			else :
				self.canUseLayerGroup = False
				break
		return
	########################################################################
	def isCurrentIsLayerGroup( self ) :
		if not pdb['gimp-image-is-valid']( self.image ) :
			return False
		if not pdb['gimp-item-is-group']( self.image.active_layer ) :
			return False
		else :
			return True
	########################################################################
	def isNormalLayer( self, target = None ) :
		if target is None :
			return False
		elif pdb['gimp-item-is-group']( target ) :
			return False
		elif pdb['gimp-item-is-text-layer']( target ) :
			return False
		else :
			return pdb['gimp-item-is-layer']( target )
	########################################################################
	def countLayers( self, groupLayer = None, recursiveLayerGroup = True ) :
		if self.image is None :
			return 0

		if groupLayer is None :
			groupLayer = self.image.active_layer

		if not pdb['gimp-item-is-group']( groupLayer ) :
			return 1

		( numChildren, childIds ) = pdb['gimp-item-get-children']( groupLayer )
		if numChildren < 1 :
			return 1

		count = 0
 
		for itemId in childIds :
			item = gimp.Item.from_id( itemId )
			if pdb['gimp-item-is-group']( item ) :
				count += self.countLayers( item, recursiveLayerGroup )
			else :
				count += 1

		return count + 1
	########################################################################
	def processCountUp( self ) :
		if self.processCount > self.layuersCount :
			value = 1.0
		else :
			value =  float( self.processCount ) / float( self.layuersCount )
		
		pdb['gimp-progress-update']( value )
	########################################################################
	def clip( self, groupLayer = None, recursiveLayerGroup = True, skipInvisible = True, mergeLayerMask = True, processCount = 0 ) :
		if not pdb['gimp-image-is-valid']( self.image ) :
			return False

		if groupLayer is None :
			groupLayer = self.image.active_layer
			self.layuersCount = self.countLayers( groupLayer, recursiveLayerGroup )
			self.processCount = 0

		if not pdb['gimp-item-is-group']( groupLayer ) :
			return False

		# グループ自体が不可視な場合はスキップ
		if skipInvisible and not pdb['gimp-item-get-visible']( groupLayer ) :
			self.processCount += 1
			return True
		
		( numChildren, childIds ) = pdb['gimp-item-get-children']( groupLayer )

		if numChildren < 1 :
			return True

		if recursiveLayerGroup :
			for itemId in childIds :
				item = gimp.Item.from_id( itemId )
				if pdb['gimp-item-is-group']( item ) :
					self.clip( item, recursiveLayerGroup, skipInvisible, mergeLayerMask )
					self.processCountUp()

		# 末尾が通常レイヤーではない場合はスキップ
		lastLayer = gimp.Item.from_id( childIds[ numChildren - 1 ] )
		if not self.isNormalLayer( lastLayer ) :
			self.processCount += numChildren
			return True

		# 末尾が不可視な場合はスキップ
		if skipInvisible and not pdb['gimp-item-get-visible']( lastLayer ) :
			self.processCount += numChildren
			return True

		if numChildren < 2 :
			self.processCount += numChildren
			return True
		
		# 選択範囲を解除
		pdb['gimp-selection-none']( self.image )

		if pdb['gimp-drawable-has-alpha']( lastLayer ) :
			# アルファマスクを選択
			pdb['gimp-image-select-item']( self.image, CHANNEL_OP_REPLACE, lastLayer )
			
		if mergeLayerMask and lastLayer.mask is not None :
			if clipMask is None :
				# レイヤーマスクを選択範囲に
				pdb['gimp-image-select-item']( self.image, CHANNEL_OP_REPLACE, lastLayer.mask )
			else :
				# レイヤーマスクとアルファマスクを合成を選択範囲に
				pdb['gimp-image-select-item']( self.image, CHANNEL_OP_INTERSECT, lastLayer.mask )

		if pdb['gimp-selection-is-empty']( self.image ) :
			# クリップすべき要素が無い
			self.processCount += numChildren
			return True

		clipMask = None

		# 選択範囲を保存
		clipMask = pdb['gimp-selection-save']( self.image )

		# グループ末尾のレイヤーは除く
		targetIdList = childIds[: numChildren - 1 ]
		########################################################################
		for itemId in targetIdList :

			self.processCount += 1
			self.processCountUp()

			layer = gimp.Item.from_id( itemId )
			if not self.isNormalLayer( layer ) :
				# 通常レイヤーではないのでスキップ
				continue

			# 選択範囲を解除
			pdb['gimp-selection-none']( self.image )

			savedLayerMask = None

			if layer.mask is not None :
				# レイヤーマスクを選択
				pdb['gimp-image-select-item']( self.image, CHANNEL_OP_REPLACE, layer.mask )

				# レイヤーマスクの選択範囲を保存
				savedLayerMask = pdb['gimp-selection-save']( self.image )

				if mergeLayerMask :
					# レイヤーマスクを反映
					pdb['gimp-layer-remove-mask']( layer, MASK_APPLY )
				else :
					# レイヤーマスクを破棄
					pdb['gimp-layer-remove-mask']( layer, MASK_DISCARD )

				# レイヤーマスクとクリップマスクの合成を選択範囲に
				pdb['gimp-image-select-item']( self.image, CHANNEL_OP_INTERSECT, clipMask )
			else :
				# クリップマスクを選択
				pdb['gimp-image-select-item']( self.image, CHANNEL_OP_REPLACE, clipMask )

			# 選択範囲からレイヤーマスクを設定
			tempLayerMask = pdb['gimp-layer-create-mask']( layer, ADD_SELECTION_MASK )

			# レイヤーマスクを設定
			pdb['gimp-layer-add-mask']( layer, tempLayerMask )

			# レイヤーマスクを反映（クリッピング）
			pdb['gimp-layer-remove-mask']( layer, MASK_APPLY )

			#pdb['gimp-image-remove-channel']( self.image, tempLayerMask )

			if savedLayerMask is not None :
				# レイヤーマスクを選択
				pdb['gimp-image-select-item']( self.image, CHANNEL_OP_REPLACE, savedLayerMask )
				# 選択範囲からレイヤーマスクを設定
				tempLayerMask = pdb['gimp-layer-create-mask']( layer, ADD_SELECTION_MASK )
				# レイヤーマスクを設定
				pdb['gimp-layer-add-mask']( layer, tempLayerMask )
				pdb['gimp-image-remove-channel']( self.image, savedLayerMask )

		########################################################################
		# 選択範囲を解除
		pdb['gimp-selection-none']( self.image )
		########################################################################
		if clipMask is not None :
			pdb['gimp-image-remove-channel']( self.image, clipMask )
########################################################################
def ExtnedOpenRasterLoad( pathName, doLoadSvgAsVectors, doCleanupEmptyGroup, doMoveParentSingleItemInGroup, doMoveParentSourceAtopItemInGroup ) :
	if not pathName :
		pdb['gimp-message']( "OpenRaster (.ora) ファイルが指定されていません。" )
		return
	elif not pathName.lower().endswith('.ora') :
		pdb['gimp-message']( "OpenRaster (.ora) ファイル以外が指定されています。" )
		return
		
	oraTool = OpenRasterTool()
	oraTool.doLoadSvgAsVectors = doLoadSvgAsVectors
	oraTool.doCleanupEmptyGroup = doCleanupEmptyGroup
	oraTool.doMoveParentSingleItemInGroup = doMoveParentSingleItemInGroup
	oraTool.doMoveParentSourceAtopItemInGroup = doMoveParentSourceAtopItemInGroup

	image = oraTool.load( pathName )
	if not isinstance( image, gimp.Image ) :
		pdb['gimp-message']( pathName + " の読み込みに失敗しました。" )
		return
 
	pdb['gimp-display-new']( image )

	return
########################################################################
def ExtnedOpenRasterSave( image, drawable, fileName, outDirectory, doSaveLayerMask, doSaveVectors, allowOverwrite ) :
	if fileName is None or str( fileName ) < 1 :
		if pdb['gimp-image-get-filename']( image ) :
			fileName = os.path.basename( pdb['gimp-image-get-filename']( image ) )
		else :
			pdb['gimp-message']( "OpenRaster (.ora) ファイルが指定されていません。" )
			return
	if not outDirectory :
		if pdb['gimp-image-get-filename']( image ) :
			outDirectory = os.path.dirname( pdb['gimp-image-get-filename']( image ) )
		else :
			pdb['gimp-message']( "保存先が指定されていません。" )
			return
	if not os.path.isdir( outDirectory ) :
		pdb['gimp-message']( "保存先が存在しません。" )
		return
	if not os.path.exists( outDirectory ) :
		pdb['gimp-message']( "保存先が存在しません。" )
		return

	if not fileName.lower().endswith('.ora') :
		fileName += '.ora'

	pathName = os.path.join( outDirectory, fileName )
	if os.path.exists( pathName ) :
		if allowOverwrite :
			os.remove( pathName )
		else :
			pdb['gimp-message']( "[" + pathName + "] ファイルが既に存在します。" )
			return
	
	oraTool = OpenRasterTool()

	oraTool.doSaveLayerMask = doSaveLayerMask
	oraTool.doSaveVectors = doSaveVectors
	
	if not oraTool.save( image, drawable, pathName ) :
		pdb['gimp-message']( "保存に失敗しました。" )
		
	return
########################################################################
def ClipLastLayer( image, drawable, recursiveLayerGroup, mergeLayerMask, skipInvisibleLayer ) :

	if image is None :
		pdb['gimp-message']( "イメージが設定されていません。" )
		return
	if not pdb['gimp-image-is-valid']( image ) :
		pdb['gimp-message']( "イメージが不正です。" )
		return

	clip = LayserClip( image )

	if not clip.isCurrentIsLayerGroup() :
		pdb['gimp-message']( "レイヤーグループが選択されていません。" )
		return
	
	pdb['gimp-image-undo-group-start']( image )
	clip.clip( None, recursiveLayerGroup, mergeLayerMask, skipInvisibleLayer )
	pdb['gimp-progress-update']( 1.0 )
	pdb['gimp-image-undo-group-end']( image )

	return
########################################################################
register(
	'LoadExtenedOpenRasterThumbnail',		# プロシジャの名前
	'OpenRaster のサムネイルを読み込む (.ora)',	# プロシジャの説明文
	'OpenRaster のサムネイルを読み込む (.ora)',	# PDBに登録する追加情報
	'かんら・から',					# 作者名
	'かんら・から　GPLv3',				# ライセンス情報
	'2014.05.31',					# 作成日
	None,						# メニューアイテム
	None,						# 対応する画像タイプ
	[	#input args. Format ( type, name, description, default [, extra] )
		( PF_STRING, 'filename', 'The name of the file to load', '' ),
		( PF_INT, 'thumb-size', 'Preferred thumbnail size', None ),
	],	# プロシジャの引数
	[
		( PF_IMAGE, 'image', 'Thumbnail image' ),
		( PF_INT, 'image-width', 'Width of full-sized image' ),
		( PF_INT, 'image-height', 'Height of full-sized image' )
	],	#戻り値の定義 Format ( type, name, description )
	ExtnedOpenRasterThumbnailer			# 処理を埋け持つ関数名
)
########################################################################
register(
	'LoadExtenedOpenRaster',			# プロシジャの名前
	'拡張された OpenRaster の読み込み。(.ora)',	# プロシジャの説明文
	'ver 2.8 以上を対象としたOpenRaster の読み込みクリプト。グループレイヤーなどに対応。',
	# PDBに登録する追加情報
	'かんら・から',					# 作者名
	'かんら・から　GPLv3',				# ライセンス情報
	'2015.05.12',					# 作成日
	'OpenRaster　拡張された形式の読み込み',		# メニューアイテム
	None,						# 対応する画像タイプ
	[	#input args. Format ( type, name, description, default [, extra] )
		( PF_FILE, 'pathName', '対称のファイルを指定して下さい。(.ora)', '' ),
		( PF_BOOL, 'doLoadSvgAsVectors', 'SVGをパスとして読み込み', True ),
		( PF_BOOL, 'doCleanupEmptyGroup', '空のグループを削除', True ),
		( PF_BOOL, 'doMoveParentSingleItemInGroup', 'グループ内に1項目しか存在しない場合は上位へ移動', False ),
		( PF_BOOL, 'doMoveParentSourceAtopItemInGroup', 'グループに Source Atop が設定されている場合は上位へ移動', True ),
	],	# プロシジャの引数
	[],	#戻り値の定義 Format ( type, name, description )
	ExtnedOpenRasterLoad,			# 処理を埋け持つ関数名
	menu = '<Toolbox>/File/Create/拡張読み込み',# メニュー表示場所
	)
########################################################################
register(
	'SaveExtenedOpenRaster',			# プロシジャの名前
	'拡張された OpenRaster の保存。(.ora)',	# プロシジャの説明文
	'ver 2.8 以上を対象としたOpenRaster の保存クリプト。グループレイヤーなどに対応。',
	# PDBに登録する追加情報
	'かんら・から',					# 作者名
	'かんら・から　GPLv3',				# ライセンス情報
	'2014.05.31',					# 作成日
	'OpenRaster 拡張された形式での保存',		# メニューアイテム
	'*',						# 対応する画像タイプ
	[	#input args. Format ( type, name, description, default [, extra] )
		( PF_IMAGE, 'image', 'Input image', None ),
		( PF_DRAWABLE, 'drawable', 'Input drawable', None ),
		( PF_STRING, 'fileName', 'ファイル名を指定して下さい。(.ora)\n指定されない場合は、イメージの名前が利用されます。', '' ),
		( PF_DIRNAME, "outDirectory", '保存先を指定して下さい。\n指定されない場合は、イメージの格納先が利用されます。', None ),
		( PF_BOOL, 'doSaveLayerMask', 'レイヤーマスクを保存', False ),
		( PF_BOOL, 'doSaveVectors', 'パスを保存', True ),
		( PF_BOOL, 'allowOverwrite', 'ファイルの上書きを許可', False ),
	],	# プロシジャの引数
	[],	#戻り値の定義 Format ( type, name, description )
	ExtnedOpenRasterSave,			#　処理を埋け持つ関数名
	menu = '<Image>/File/Save/拡張書き込み',	# メニュー表示場所
)
########################################################################
register(
	'ClipLastLayer',				# プロシジャの名前
	'レイヤーグループの最下レイヤーでクリップする。',	# プロシジャの説明文
	'ver 2.8 以上を対象としたクリップクリプト。レイヤーグループが選択されている場合、Krita の様な最下レイヤーでクリッピングを行う。',
	# PDBに登録する追加情報
	'かんら・から',					# 作者名
	'GPLv3',					# ライセンス情報
	'2015.03.19',					# 作成日
	'レイヤーグループクリップ(Python)',		# メニューアイテム
	'*',						# 対応する画像タイプ
	[	#input args. Format ( type, name, description, default [, extra] )
		(PF_IMAGE, 'image', 'Input image', None),
		(PF_DRAWABLE, 'drawable', 'Input drawable', None),
		(PF_BOOL, 'recursiveLayerGroup', '再帰的にレイヤーグループを走査する:', 1),
		(PF_BOOL, 'mergeLayerMask', 'レイヤーマスクを参照する:', 1),
		(PF_BOOL, 'skipInvisibleLayer', '非表示レイヤーをスキップする:', 1)
	],	# プロシジャの引数
	[],	# 戻り値の定義
	ClipLastLayer,				# 処理を埋け持つ関数名
	menu='<Image>/Layer/レイヤー操作(Python-fu)'	# メニュー表示場所
	)
########################################################################

main() # プラグインを駆動させるための関数

