#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# firewalld_ipblock.py version 2.3.1
# Copyright (C) 2015 かんら・から  twitter:@kanrakara 
# 
# ipblock is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# ipblock is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

import urllib.parse
import urllib.request, urllib.parse, urllib.error
import urllib.request, urllib.error, urllib.parse
import re
import subprocess
import shlex
import os
import sys
import argparse
import xml.dom.minidom
import time

#########################################################################
# start class FirewallDirectXML
#########################################################################
class FirewallDirectXML :
	#########################################################################
	document = None
	implementation = None
	topElement = None
	#########################################################################
	def __init__( self ) :
		self.implementation = xml.dom.minidom.getDOMImplementation()
		self.document = self.implementation.createDocument( None, 'direct', None )
		self.topElement = self.document.documentElement
	#########################################################################
	def toXmlString( self, reform = True ) :
		if not self.document :
			return ''
		if not self.topElement :
			return ''
		
		if not reform :
			return self.document.toxml('utf-8')

		xmlStr = str( self.document.toxml('utf-8'), 'utf-8' )

		xmlStr = xmlStr.replace( '<direct', '\n<direct' )
		xmlStr = xmlStr.replace( '</direct', '\n</direct' )
		xmlStr = xmlStr.replace( '<chain', '\n    <chain' )
		xmlStr = xmlStr.replace( '<rule', '\n    <rule' )
		xmlStr = xmlStr.replace( '<passthrough', '\n    <passthrough' )

		return xmlStr
	#########################################################################
	def createChain( self, chainName, ipv = 'ipv4', table = 'filter' ) :
		if not isinstance( chainName, str ) :
			return None
		if not isinstance( ipv, str ) :
			return None
		if not isinstance( table, str ) :
			return None
		if not self.document :
			return None
		if not self.topElement :
			return None

		chainName = chainName.strip()
		ipv = ipv.strip().lower()
		table = table.strip()

		if len( chainName ) < 1 :
			return None
		if len( ipv ) < 1 :
			return None
		if len( table ) < 1 :
			return None
		if ipv not in [ 'ipv4', 'ipv6' ] :
			return None

		element = self.document.createElement( 'chain' )
		element.setAttribute( 'ipv', ipv )
		element.setAttribute( 'table', table )
		element.setAttribute( 'chain', chainName )

		self.topElement.appendChild( element )

		return element
	#########################################################################
	def createRule( self, optionStr, chainName, ipv = 'ipv4', priority = 1, table = 'filter' ) :
		if not isinstance( optionStr, str ) :
			return None
		if not isinstance( chainName, str ) :
			return None
		if not isinstance( ipv, str ) :
			return None
		if not isinstance( table, str ) :
			return None
		if not isinstance( priority, int ) :
			return None
		if not self.document :
			return None
		if not self.topElement :
			return None

		optionStr = optionStr.strip()
		chainName = chainName.strip()
		ipv = ipv.strip().lower()
		table = table.strip()

		if len( optionStr ) < 1 :
			return None
		if len( ipv ) < 1 :
			return None
		if len( table ) < 1 :
			return None
		if ipv not in [ 'ipv4', 'ipv6' ] :
			return None

		element = self.document.createElement( 'rule' )
		element.setAttribute( 'ipv', ipv )
		element.setAttribute( 'table', table )
		element.setAttribute( 'chain', chainName )
		element.setAttribute( 'priority', str( priority ) )
		
		textElelement = self.document.createTextNode( optionStr )
		element.appendChild( textElelement )

		self.topElement.appendChild( element )
		
		return element
	#########################################################################
	def createPassthrough( self, optionStr, ipv = 'ipv4' ) :
		if not isinstance( optionStr, str ) :
			return None
		if not isinstance( ipv, str ) :
			return None
		if not self.document :
			return None
		if not self.topElement :
			return None

		optionStr = optionStr.strip()
		ipv = ipv.strip().lower()

		if len( optionStr ) < 1 :
			return None
		if len( ipv ) < 1 :
			return None
		if ipv not in [ 'ipv4', 'ipv6' ] :
			return None

		element = self.document.createElement( 'passthrough' )
		element.setAttribute( 'ipv', ipv )

		textElelement = self.document.createTextNode( optionStr )
		element.appendChild( textElelement )

		self.topElement.appendChild( element )

		return element
#########################################################################
# end class FirewallDirectXML
#########################################################################

#########################################################################
# start class IpCountries
#########################################################################
class IpCountries:
	#########################################################################
	errorMessages = []
	dataSource = []
	countories = []
	ipV4File = None
	ipV6File = None
	curCountry = ''
	ipv4FilePath = ''
	ipv6FilePath = ''
	ipv4DataLength = 0
	ipv6DataLength = 0
	ipv4spltLengthList = []
	#########################################################################
	def __init__( self ) :
		self.reset()

		self.ipv4spltLengthList = []
		for shiftBit in range( 32 ) :
			self.ipv4spltLengthList.append( 1 << shiftBit )
		self.ipv4spltLengthList.reverse()
	#########################################################################
	def __del__( self ) :
		if self.ipV4File :
			self.ipV4File.close()
		if self.ipV6File :
			self.ipV6File.close()
	#########################################################################
	def reset( self ) :
		##############################################################
		if self.ipV4File :
			self.ipV4File.close()
		if self.ipV6File :
			self.ipV6File.close()
		##############################################################
		self.errorMessages = []
		self.dataSource = []
		self.countories = []
		self.ipV4File = None
		self.ipV6File = None
		self.curCountry = ''
		self.ipv4FilePath = ''
		self.ipv6FilePath = ''
		self.ipv4DataLength = 0
		self.ipv6DataLength = 0	
		##############################################################
		
	#########################################################################
	def hasError( self ) :
		return len( self.errorMessages ) > 0
	#########################################################################
	def getErrorMessages( self ) :
		return self.errorMessages
	#########################################################################
	def addDataSource( self, url ):
		##############################################################
		if not isinstance( url, str ) :
			return False
		##############################################################
		added = False
		urlList = re.split( '\r\n|\n', url )
		for urlRow in urlList :
			##############################################################
			urlRow = urlRow.strip()
			urlParsed = urllib.parse.urlparse( urlRow )
			##############################################################
			if not urlParsed.path :
				continue
			if urlRow in self.dataSource :
				continue
			##############################################################
			self.dataSource.append( urlRow )
			added = True
		##############################################################
		return added
	#########################################################################
	def addCountries( self, country ) :
		##############################################################
		if not isinstance( country, str ) :
			return False
		##############################################################
		countryList = re.split( '\r\n|\n', country )
		for urlRow in countryList :
			##############################################################
			countryRow = urlRow.strip().upper()
			if len( countryRow ) != 2 :
				continue
			##############################################################
			if countryRow in self.countories :
				continue
			##############################################################
			self.countories.append( countryRow )
			added = True
		##############################################################
		return added
	#########################################################################
	def getCountries( self ) :
		return self.countories
	#########################################################################
	def getIPv4DataCount( self ) :
		return self.ipv4DataLength
	#########################################################################
	def getIPv6DataCount( self ) :
		return self.ipv6DataLength
	#########################################################################
	def loadData( self, ipv4Path, ipv6Path, cacheMaxDay = 1 ) :
		##############################################################
		if not isinstance( ipv4Path, str ) :
			return False
		if not isinstance( ipv6Path, str ) :
			return False
		##############################################################
		if os.path.isfile( ipv4Path ) and os.path.isfile( ipv6Path ) :

			ipv4PastSec = time.time() - os.path.getctime( ipv4Path )
			ipv6PastSec = time.time() - os.path.getctime( ipv6Path )
			cacheMaxDSec = cacheMaxDay * 24 * 60 * 60

			if ipv4PastSec < cacheMaxDSec and ipv6PastSec < cacheMaxDSec :
				##############################################################
				self.ipv4FilePath = ipv4Path
				self.ipv6FilePath = ipv6Path
				self.ipv4DataLength = 0
				self.ipv6DataLength = 0
				##############################################################
				with open( self.ipv4FilePath, 'r' ) as fp :
					for line in fp :
						self.ipv4DataLength += 1
				##############################################################
				with open( self.ipv6FilePath, 'r' ) as fp :
					for line in fp :
						self.ipv6DataLength += 1
				##############################################################
				return True
		##############################################################
		try :
			##############################################################
			if self.ipV4File :
				self.ipV4File.close()
			if self.ipV6File :
				self.ipV6File.close()
			##############################################################
		except :
			##############################################################
				self.ipV4File = None
				self.ipV6File = None
			##############################################################
		try :
			##############################################################
			self.ipV4File = open( ipv4Path, 'w' )
			self.ipV6File = open( ipv6Path, 'w' )
			##############################################################
			self.ipv4FilePath = ipv4Path
			self.ipv6FilePath = ipv6Path
			self.ipv4DataLength = 0
			self.ipv6DataLength = 0	
			##############################################################
			for url in self.dataSource :
				##############################################################
				stream = None
				line = 'Open URL : ' + url
				try :
					stream = urllib.request.urlopen( url )
				except :
					self.errorMessages.append( 'URL Open error : ' + url )
					continue
				##############################################################
				for line in stream :
					##############################################################
					# Use Extened Format
					# see : http://www.apnic.net/publications/media-library/documents/resource-guidelines/rir-statistics-exchange-format
					##############################################################
					if isinstance( line, bytes ) :
						line = line.decode('utf-8')
					##############################################################
					if len( line ) < 7 :
						continue
					if line[0] == '#' :
						continue
					##############################################################
					line = line.rstrip()
					record = line.rstrip().split( '|' )
					if len( record ) < 7 :
						continue
					##############################################################
					country = record[1]
					if country not in self.countories :
						continue
					##############################################################
					ipType = record[2].lower()
					if ipType not in [ 'ipv4', 'ipv6' ] :
						continue
					##############################################################
					ipString = record[3]
					assignLength = int ( record[4] )
					##############################################################
					if ipType == 'ipv4' :
						##############################################################
						for newIpWithMaskString in self.ipv4StrToWithMask( ipString, assignLength ) :
							##############################################################
							if len( newIpWithMaskString ) < 0 or newIpWithMaskString == '0.0.0.0'  :
								continue
							##############################################################
							self.ipV4File.write( country + '|' + newIpWithMaskString + "\n" )
							self.ipv4DataLength += 1
							##############################################################
						##############################################################
					elif ipType == 'ipv6' :
						maskStr = country + '|' + ipString + "/" + str( assignLength ) + "\n"
						if len( maskStr ) > 0 :
							self.ipV6File.write( maskStr )
							self.ipv6DataLength += 1
						else :
							self.errorMessages.append( "IPv6 Eorror : " + line )
					else :
						continue
				##############################################################
				# line end for
				##############################################################
				stream.close()
			##############################################################
			# dataSource end for
			##############################################################
			self.ipV4File.flush()
			self.ipV6File.flush()
			##############################################################
			self.ipV4File.close()
			self.ipV6File.close()
			##############################################################
			self.ipV4File = None
			self.ipV6File = None
			##############################################################
		except :
			##############################################################
			if self.ipV4File :
				self.ipV4File.close()
			if self.ipV6File :
				self.ipV6File.close()
			##############################################################
			self.ipV4File = None
			self.ipV6File = None
			##############################################################
			self.errorMessages.append( "Data Create Error : " + line )
			##############################################################
			return False
			##############################################################
		return True
	#########################################################################
	def ipv4StrToNum( self, ipv4Str ) :
		##############################################################
		if not isinstance( ipv4Str, str ) :
			return 0
		##############################################################
		ipv4PatrList = ipv4Str.split('.')
		if len( ipv4PatrList ) != 4 :
			return 0
		##############################################################
		ip = 0
		for partStr in ipv4PatrList:
			ip <<= 8
			ip |= int( partStr )
		##############################################################
		return ip
	#########################################################################
	def ipv4NumToStr( self, ipv4Num ) :
		##############################################################
		if not isinstance( ipv4Num, int ) and not isinstance( ipv4Num, int ) :
			return '0.0.0.0'
		##############################################################
		if ipv4Num < 1 :
			return '0.0.0.0'
		##############################################################
		ipv4StrList = []
		num = ipv4Num
		for i in range( 4 ) :
			ipv4StrList = [ str( num % 256 ) ] + ipv4StrList
			num /= 256

		return '.'.join( ipv4StrList )
	#########################################################################
	def ipv4StrToWithMask( self, startIPv4Str, assignLength, ipv4StrWithMaskList = None ) :
		##############################################################
		if ipv4StrWithMaskList is None :
			ipv4StrWithMaskList = []
		##############################################################
		if not isinstance( startIPv4Str, str ) :
			return ipv4StrWithMaskList
		##############################################################
		if not isinstance( assignLength, int ) and not isinstance( assignLength, int ) :
			return ipv4StrWithMaskList
		##############################################################
		if assignLength <= 1 :
			ipv4StrWithMaskList.append( startIPv4Str )
			return ipv4StrWithMaskList
		##############################################################
		if assignLength not in self.ipv4spltLengthList :
			##############################################################
			# need split
			##############################################################
			ipNum = self.ipv4StrToNum( startIPv4Str )
			if ipNum < 1 :
				return ipv4StrWithMaskList
			
			bitHit = 1
			maskLength = 32
			while not ( ipNum & bitHit ) :
				bitHit <<= 1
				maskLength -= 1

			##############################################################
			if ( 0x01 << ( 32 - maskLength ) ) < assignLength :

				ipv4StrWithMaskList.append( startIPv4Str + '/' + str( maskLength ) )
			
				ipNum += bitHit

				nextAssignLength = assignLength - bitHit
				nextIPv4Str = self.ipv4NumToStr( ipNum )

				return self.ipv4StrToWithMask( nextIPv4Str, nextAssignLength, ipv4StrWithMaskList )
			##############################################################
			for newAssignLength in self.ipv4spltLengthList :
				if newAssignLength <= assignLength :
					break
			##############################################################
			ipv4StrWithMaskList = self.ipv4StrToWithMask( startIPv4Str, newAssignLength, ipv4StrWithMaskList )
			##############################################################
			nextAssignLength = assignLength - newAssignLength
			nextIPv4Str = self.ipv4NumToStr( ipNum + newAssignLength )
			##############################################################
			return self.ipv4StrToWithMask( nextIPv4Str, nextAssignLength, ipv4StrWithMaskList )
			##############################################################
		##############################################################
		rangeMask = assignLength - 1
		ipNum = self.ipv4StrToNum( startIPv4Str )
		##############################################################
		if ipNum & rangeMask :
			##############################################################
			# Overfllow
			##############################################################
			maskLength = 32
			shitedNum = ipNum
			curAssignLength = 0x01
			while not ( shitedNum & 0x01 ) and maskLength > 0 :
				maskLength -= 1
				shitedNum >>= 1
				curAssignLength <<= 1
			
			ipv4StrWithMaskList.append( startIPv4Str + '/' + str( maskLength ) )
			
			nextAssignLength = assignLength - curAssignLength
			nextIPv4Str = self.ipv4NumToStr( ipNum + curAssignLength )

			return self.ipv4StrToWithMask( nextIPv4Str, nextAssignLength, ipv4StrWithMaskList )
			##############################################################
		else :
			##############################################################
			maskLength = 32
			bitCheck = rangeMask
			while bitCheck > 0 :
				maskLength -= 1
				bitCheck >>= 1

			ipv4StrWithMaskList.append( startIPv4Str + '/' + str( maskLength ) )

			return ipv4StrWithMaskList
			##############################################################
	#########################################################################
	def getCountryIP( self, country, ipType = 'ipv4' ) :
		##############################################################
		if not isinstance( country, str ) :
			return False
		if len( country ) != 2 :
			return False
		if not isinstance( ipType, str ) :
			return False
		if len( ipType ) != 4 :
			return False
		##############################################################
		country = country.upper()
		ipType = ipType.lower()
		##############################################################
		if country not in self.countories :
			return False
		if ipType not in [ 'ipv4', 'ipv6' ] :
			return False
		##############################################################
		try :
			##############################################################
			filePtr = None
			if ipType == 'ipv4' :
				if self.ipV4File == None :
					self.ipV4File = open( self.ipv4FilePath, 'r' )
				filePtr = self.ipV4File
			##############################################################
			elif ipType == 'ipv6' :
				if self.ipV6File == None :
					self.ipV6File = open( self.ipv6FilePath, 'r' )
				filePtr = self.ipV6File
			##############################################################
			else :
				return False
			##############################################################
			if self.curCountry != country :
				filePtr.seek(0)
				self.curCountry = country
			##############################################################
			for line in filePtr :
				line = line.strip()
				##############################################################
				if len( line ) < 1 :
					continue
				record = line.split( '|' )
				if len( record ) < 2 :
					continue
				if record[0] != country :
					continue
				if len( record[1] ) < 1 :
					continue
				##############################################################
				return record[1]
			##############################################################
			return ''
		##############################################################
		except :
			return False
		##############################################################
		return False
	#########################################################################
#########################################################################
# end class IpCountries
#########################################################################

#########################################################################
# start class ShellCommand
#########################################################################
class ShellCommand :
	#########################################################################
	regexpEnv = None
	printToStd = False
	printCommandStd=False
	nullDevice = None
	#########################################################################
	def __init__( self, printToStd = False, printCommandStd = False ) :
		self.regexpEnv = re.compile( '\\$\\{([A-Za-z_][A-Za-z0-9_]*)\\}' )
		self.printToStd = printToStd
		self.printCommandStd = printCommandStd
		if not self.printCommandStd :
			self.nullDevice = open( os.devnull, 'wb' )
	#########################################################################
	def __del__( self ):
		if self.nullDevice :
			self.nullDevice.close()
			self.nullDevice = None
	#########################################################################
	def execute( self, command, replace={} ) :
		##############################################################
		if not isinstance( command, str ) :
			return False
		if not isinstance( replace, dict ) :
			return False
		##############################################################
		command = command.strip()
		if len( command ) < 1 :
			return False

		replaceTarget = self.regexpEnv.findall( command )
		for envTarget in replaceTarget :
			if envTarget in replace :
				command = command.replace( '${' + envTarget + '}', replace[envTarget] )
			elif envTarget in os.environ :
				command = command.replace( '${' + envTarget + '}', os.environ[envTarget] )
			else :
				command = command.replace( '${' + envTarget + '}', '' )
		##############################################################
		avoidError = False
		if command.startswith( '@' ) :
			command = command[1:]
			avoidError = True
		##############################################################
		argList = []
		for arg in shlex.split( command ) :
			##############################################################
			if len( arg ) < 1 :
				continue
			##############################################################
			argList.append( arg )
		##############################################################
		if len( argList ) < 1 :
			return False
		##############################################################
		if self.printToStd :
			regexpEsc = re.compile( '[\\s\\[\\]\\{\\}\"\']+' )
			regexpSingle = re.compile( '(?<!\\\\)\'' )
			printList = []
			for arg in argList :
				if regexpEsc.search( arg ) :
					arg = regexpSingle.sub( "\\'", arg )
					arg = "'" + arg + "'"
				printList.append( arg )
					
			print(' '.join( printList ))
			return True
		##############################################################
		retCode = subprocess.call( argList, stdout=self.nullDevice )
		if retCode == 0 or avoidError:
			return True
		else :
			return False
	#########################################################################
#########################################################################
# end class ShellCommand
#########################################################################

#########################################################################
# start class ActionControl
#########################################################################
class ActionControl :
	#########################################################################
	errorMessages = []
	configFile = ''
	config = None
	configDir = ''
	lineSeparatorRegexp = None
	optionRegexp = None
	regexpEnv = None
	command = None
	forceAction = True
	reformXml = True
	skipReloadSameXml = True
	denyCountries = None
	printToStd = False
	directXml = None
	outputFile = None
	cacheMaxDay = 1
	isSameXml = False
	#########################################################################
	def __init__( self, configFile='', printToStd = False, outputFile='' ) :
		##############################################################
		if not isinstance( configFile, str ) or configFile == '' :
			 configFile = '/usr/local/etc/firewalld_ipblock/firewalld_ipblock.conf'
		if not isinstance( outputFile, str ) or outputFile == '' :
			 outputFile = '/etc/firewalld/direct.xml'
		##############################################################
		self.configFile = configFile
		self.printToStd = printToStd
		self.configDir = os.path.dirname( configFile )
		self.outputFile = outputFile
		##############################################################
		cacheMaxDay = self.getParam ( 'cacheAliveDate' )
		if cacheMaxDay.isdigit() :
			self.cacheMaxDay = int( cacheMaxDay )
		else :
			self.cacheMaxDay = 1
		##############################################################
		import configparser
		##############################################################
		self.errorMessages = []
		self.config = configparser.ConfigParser()
		self.config.read( self.configFile )
		##############################################################
		self.lineSeparatorRegexp = re.compile( '\r\n|\n' )
		self.optionRegexp = optionRegexp = re.compile( '(([\\d]+|ipv4|ipv6)\\s+)?(-t\\s+(filter|nat|mangle|raw)\\s+)?((-[AN])\\s+([A-Za-z][\\w]*)\\s*)' )
		self.regexpEnv = re.compile( '\\$\\{([A-Za-z_][A-Za-z0-9_]*)\\}' )

		##############################################################
		self.forceAction = self.getFlag ( 'forceAction' )
		self.reformXml = self.getFlag ( 'reformXml' )
		self.skipReloadSameXml = self.getFlag ( 'skipReloadSameXml' )
		##############################################################
		self.command = ShellCommand( False, False )
		##############################################################
		self.directXml = FirewallDirectXML()
		##############################################################
		self.denyCountries = IpCountries()
		try :
			self.denyCountries.addDataSource( self.config.get( 'dataSource', 'url' ) )
			self.denyCountries.addCountries( self.config.get( 'countries', 'deny' ) )
		except :
			self.denyCountries.reset()
	#########################################################################
	def getFlag( self, option ) :
		##############################################################
		if not isinstance( option, str ) :
			return False
		##############################################################
		try :
			flag = self.config.get( 'actionFlag', option )
			flag = flag.lower()
			if flag == 'false' or flag == 'deny' or flag == '0' :
				return False
			else :
				return True
		except :
			return False
	#########################################################################
	def getParam( self, option ) :
		##############################################################
		if not isinstance( option, str ) :
			return ''
		##############################################################
		try :
			flag = self.config.get( 'actionParam', option )
			return flag.strip()
		except :
			return ''
	#########################################################################
	def getOption( self, option ) :
		##############################################################
		if not isinstance( option, str ) :
			return []
		##############################################################
		try :
			optionLines = self.config.get( 'option', option )
			optionList = []
			for line in self.lineSeparatorRegexp.split( optionLines ) :
				line = line.strip()
				if len( line ) < 1 :
					continue
				else :
					optionList.append( line )
			return optionList
		except :
			return []
	#########################################################################
	def getCommand( self, option ) :
		##############################################################
		if not isinstance( option, str ) :
			return []
		##############################################################
		try :
			commandLines = self.config.get( 'command', option )
			commandList = []
			for line in self.lineSeparatorRegexp.split( commandLines ) :
				line = line.strip()
				if len( line ) < 1 :
					continue
				else :
					commandList.append( line )
			return commandList
		except :
			return []
	#########################################################################
	def parseIptablesOption( self, option ) :
		##############################################################
		if not isinstance( option, str ) :
			return {}
		##############################################################
		option = option.strip()
		##############################################################
		if len( option ) < 1 :
			return {}
		##############################################################
		retDic = { 'priority' : None, 'table' : None, 'chain' : None, 'newChain' : False, 'addRule' : False, 'option' : '' }
		
		match = self.optionRegexp.match( option )
		if not match :
			retDic['option'] = option
		else :
			retDic['priority'] = int( match.group( 2 ) ) if match.group( 2 ).isdigit() else None
			retDic['table'] = match.group( 4 ) if match.group( 4 ) else 'filter'
			retDic['chain'] = match.group( 7 ) if match.group( 7 ) else None
			retDic['newChain'] = True if '-N' == match.group( 6 ) else False
			retDic['addRule'] = True if '-A' == match.group( 6 ) else False
			retDic['option'] = self.optionRegexp.sub( '', option )
			retDic['ipv'] = match.group( 2 ) if match.group( 2 ) in [ 'ipv4', 'ipv6' ] else None
			
		return retDic
	#########################################################################
	def doOption( self, optionList, ipv = 'ipv4' ) :
		##############################################################
		if not isinstance( optionList, list ) :
			return False
		##############################################################
		if len( optionList ) < 1 :
			return True
		##############################################################
		if not isinstance( ipv, str ) :
			return False
		##############################################################
		if ipv not in [ 'ipv4', 'ipv6' ] :
			return False
		##############################################################
		for line in optionList :
			optionDic = self.parseIptablesOption( line )
			if optionDic == {} :
				continue

			if optionDic['newChain'] :
				if self.directXml.createChain( optionDic['chain'], ipv, optionDic['table'] ) is None :
					self.errorMessages.append( 'BAD Option Error :' + line )
					if not self.forceAction :
						return False
					else :
						continue
			elif optionDic['addRule'] :

				if self.directXml.createRule( optionDic['option'], optionDic['chain'], ipv, optionDic['priority'], optionDic['table'] ) is None :
					self.errorMessages.append( 'BAD Option Error :' + line )
					if not self.forceAction :
						return False
					else :
						continue
			else :
				if optionDic['ipv'] in  [ 'ipv4', 'ipv6' ] :
					ipv = optionDic['ipv']
				
				if self.directXml.createPassthrough( optionDic['option'], ipv ) is None :
					self.errorMessages.append( 'BAD Option Error :' + line )
					if not self.forceAction :
						return False
					else :
						continue
		##############################################################
		return True
	#########################################################################
	def doLoopBackTcp( self ) :
		##############################################################
		if not self.doOption( self.getOption( 'ipv4Loopback' ), 'ipv4' ) :
			return False
		##############################################################
		if not self.doOption( self.getOption( 'ipv6Loopback' ), 'ipv6' ) :
			return False
		##############################################################
		return True
	#########################################################################
	def doNewNotSynTcp( self ) :
		##############################################################
		if not self.doOption( self.getOption( 'ipv4NewNotSynTcp' ), 'ipv4' ) :
			return False
		##############################################################
		if not self.doOption( self.getOption( 'ipv6NewNotSynTcp' ), 'ipv6' ) :
			return False
		##############################################################
		return True
	#########################################################################
	def doNewSynAckTcp( self ) :
		##############################################################
		if not self.doOption( self.getOption( 'ipv4NewSynAckTcp' ), 'ipv4' ) :
			return False
		##############################################################
		if not self.doOption( self.getOption( 'ipv6NewSynAckTcp' ), 'ipv6' ) :
			return False
		##############################################################
		return True
	#########################################################################
	def doDropInvalid( self ) :
		##############################################################
		if not self.doOption( self.getOption( 'ipv4DropInvalid' ), 'ipv4' ) :
			return False
		##############################################################
		if not self.doOption( self.getOption( 'ipv6DropInvalid' ), 'ipv6' ) :
			return False
		##############################################################
		return True
	#########################################################################
	def doDropBadTcp( self ) :
		##############################################################
		if not self.doOption( self.getOption( 'ipv4DropBadTcp' ), 'ipv4' ) :
			return False
		##############################################################
		if not self.doOption( self.getOption( 'ipv6DropBadTcp' ), 'ipv6' ) :
			return False
		##############################################################
		return True
	#########################################################################
	def doSpecialAction( self, dataFilePath, ipv = 'ipv4', startOptionName = None, targetOptionName = None , endOptionName = None ) :
		##############################################################
		if not self.doOption( self.getOption( startOptionName ), ipv ) :
			return False
		##############################################################
		try :
			filePathList = self.config.get( 'files', dataFilePath )
			for filePath in self.lineSeparatorRegexp.split( filePathList ) :

				filePath = filePath.strip()
				filePath = os.path.join( self.configDir, filePath )
				if not os.path.exists( filePath ) :
					self.errorMessages.append( 'doSpecialAction Load Error : ' + filePath )
					if not self.forceAction :
						return False
					else :
						continue

				optionList = self.getOption( targetOptionName )
				if len( optionList ) < 1 :
					continue
			
				with open( filePath, 'r' ) as fp :
				##############################################################
					for address in fp :
						##############################################################
						address = address.strip()
						if len( address ) < 1 :
							continue
						elif address[0] == '#' :
							continue
						##############################################################
						replace = { 'TargetAddress' : address }
						##############################################################
						for line in optionList :
							##############################################################
							replaceTarget = self.regexpEnv.findall( line )
							for envTarget in replaceTarget :
								if envTarget in replace :
									line = line.replace( '${' + envTarget + '}', replace[envTarget] )
								elif envTarget in os.environ :
									line = line.replace( '${' + envTarget + '}', os.environ[envTarget] )
								else :
									line = line.replace( '${' + envTarget + '}', '' )
							##############################################################
							if not self.doOption( [ line ] ) :
								if not self.forceAction :
									return False
								else :
									continue
						##############################################################
				##############################################################
			##############################################################
			if not self.doOption( self.getOption( endOptionName ), ipv ) :
				return False
			##############################################################
			return True
			##############################################################
		except :
			return False
	#########################################################################
	def doSpecialAllowInputIPv4( self ) :
		return self.doSpecialAction( 'dataAllowFileIPv4', 'ipv4', 'ipv4SpecialIpAcceptStart', 'ipv4SpecialIpAcceptTarget', 'ipv4SpecialIpAcceptEnd' )
	#########################################################################
	def doSpecialAllowInputIPv6( self ) :
		return self.doSpecialAction( 'dataAllowFileIPv6', 'ipv6', 'ipv6SpecialIpAcceptStart', 'ipv6SpecialIpAcceptTarget', 'ipv6SpecialIpAcceptEnd' )
	#########################################################################
	def doSpecialDenyInputIPv4( self ) :
		return self.doSpecialAction( 'dataDenyFileIPv4', 'ipv4', 'ipv4SpecialIpDropStart', 'ipv4SpecialIpDropTarget', 'ipv4SpecialIpDropEnd' )
	#########################################################################
	def doSpecialDenyInputIPv6( self ) :
		return self.doSpecialAction( 'dataDenyFileIPv6', 'ipv6', 'ipv6SpecialIpDropStart', 'ipv6SpecialIpDropTarget', 'ipv6SpecialIpDropEnd' )
	#########################################################################
	def loadDenyCountries( self, forceReload = False ) :
		##############################################################
		if self.denyCountries is None :
			self.denyCountries = IpCountries()
		##############################################################
		if forceReload is False :
			if self.denyCountries.getIPv4DataCount() > 0 :
				return True
			if self.denyCountries.getIPv6DataCount() > 0 :
				return True
		##############################################################
		try :
			#########################################################################
			self.denyCountries.reset()
			#########################################################################
			self.denyCountries.addDataSource( self.config.get( 'dataSource', 'url' ) )
			self.denyCountries.addCountries( self.config.get( 'countries', 'deny' ) )
			#########################################################################
			dataFileIPv4 = self.config.get( 'files', 'dataFileIPv4' )
			dataFileIPv6 = self.config.get( 'files', 'dataFileIPv6' )
			#########################################################################
			results = self.denyCountries.loadData( dataFileIPv4, dataFileIPv6, self.cacheMaxDay )
			if not results :
				return False
			elif not self.forceAction and self.denyCountries.hasError() :
				return False
			#########################################################################
			return True
			#########################################################################
		except :
			return False
	#########################################################################
	def doDenyCountryIP( self, startOption, denyOption, endOption, country, ipType ) :
		##############################################################
		if not isinstance( startOption, list ) :
			return False
		##############################################################
		if not isinstance( denyOption, list ) :
			return False
		##############################################################
		if not isinstance( endOption, list ) :
			return False
		##############################################################
		if not isinstance( country, str ) :
			return False
		##############################################################
		if not isinstance( ipType, str ) :
			return False
		ipType = ipType.lower()
		if ipType not in [ 'ipv4', 'ipv6' ] :
			return False
		##############################################################
		if len( denyOption ) > 0 :
			##############################################################
			if not self.loadDenyCountries() :
				##############################################################
				self.errorMessages.append( 'loadDenyCountries Error ' )
				if self.denyCountries is not None :
					for line in self.denyCountries.getErrorMessages() :
						self.errorMessages.append( 'loadDenyCountries Error : ' + line )
				if not self.forceAction :
					return False
				##############################################################
			##############################################################
			elif country not in self.denyCountries.getCountries() :
				return False
		##############################################################
		for line in startOption :
			##############################################################
			replace = { 'TargetCountry' : country }
			##############################################################
			replaceTarget = self.regexpEnv.findall( line )
			for envTarget in replaceTarget :
				if envTarget in replace :
					line = line.replace( '${' + envTarget + '}', replace[envTarget] )
				elif envTarget in os.environ :
					line = line.replace( '${' + envTarget + '}', os.environ[envTarget] )
				else :
					line = line.replace( '${' + envTarget + '}', '' )
			##############################################################
			if not self.doOption( [ line ], ipType ) :
				if not self.forceAction :
					return False
				else :
					continue
		##############################################################
		while self.denyCountries is not None :
			ip = self.denyCountries.getCountryIP( country, ipType )
			##############################################################
			if ip is False :
				# Error
				self.errorMessages.append( 'getCountryIP Error :' + country + ':' + ipType )
				if not self.forceAction :
					return False
				else :
					continue
			elif len( ip ) < 1 :
				# End of Data
				break
			##############################################################
			replace = { 'TargetCountry' : country, 'TargetAddress' : ip }
			for line in denyOption :
				##############################################################
				replaceTarget = self.regexpEnv.findall( line )
				for envTarget in replaceTarget :
					if envTarget in replace :
						line = line.replace( '${' + envTarget + '}', replace[envTarget] )
					elif envTarget in os.environ :
						line = line.replace( '${' + envTarget + '}', os.environ[envTarget] )
					else :
						line = line.replace( '${' + envTarget + '}', '' )
				##############################################################
				if not self.doOption( [ line ], ipType ) :
					if not self.forceAction :
						return False
					else :
						continue
			##############################################################
		##############################################################
		for line in endOption :
			##############################################################
			replace = { 'TargetCountry' : country }
			##############################################################
			replaceTarget = self.regexpEnv.findall( line )
			for envTarget in replaceTarget :
				if envTarget in replace :
					line = line.replace( '${' + envTarget + '}', replace[envTarget] )
				elif envTarget in os.environ :
					line = line.replace( '${' + envTarget + '}', os.environ[envTarget] )
				else :
					line = line.replace( '${' + envTarget + '}', '' )
			##############################################################
			if not self.doOption( [ line ], ipType ) :
				if not self.forceAction :
					return False
				else :
					continue
		##############################################################
		return True
	#########################################################################
	def doDenyIPv4Countries( self ) :
		##############################################################
		if self.denyCountries is None :
			return False
		##############################################################
		dropIPv4Start = self.getOption( 'ipv4DorpTargetCountryStart' )
		dropIPv4Option = self.getOption( 'ipv4DorpTargetCountry' )
		dropIPv4End = self.getOption( 'ipv4DorpTargetCountryEnd' )
		##############################################################
		for country in self.denyCountries.getCountries() :
			if not self.doDenyCountryIP( dropIPv4Start, dropIPv4Option, dropIPv4End, country, 'ipv4' ) :
				if not self.forceAction :
					return False
				else :
					continue
		##############################################################
		return True
	#########################################################################
	def doDenyIPv6Countries( self ) :
		##############################################################
		if self.denyCountries is None :
			return False
		##############################################################
		dropIPv6Start = self.getOption( 'ipv6DorpTargetCountryStart' )
		dropIPv6Option = self.getOption( 'ipv6DorpTargetCountry' )
		dropIPv6End = self.getOption( 'ipv6DorpTargetCountryEnd' )
		##############################################################
		for country in self.denyCountries.getCountries() :
			if not self.doDenyCountryIP( dropIPv6Start, dropIPv6Option, dropIPv6End, country, 'ipv6' ) :
				if not self.forceAction :
					return False
				else :
					continue
		##############################################################
		return True
	#########################################################################
	def doSave( self ) :
		##############################################################
		if self.printToStd :
			print(self.directXml.toXmlString())
			return True
		##############################################################
		try :
			##############################################################
			fp = None
			##############################################################
			if self.skipReloadSameXml :
				self.isSameXml = False

				with open( self.outputFile, 'r', encoding = 'utf-8' ) as fp :
					orgXmlStr = fp.read()
					self.isSameXml = ( orgXmlStr == self.directXml.toXmlString() )

				fp = None

				if self.isSameXml :
					# direct.xml is same. skip writing
					return True
			##############################################################
			with open( self.outputFile, 'w', encoding = 'utf-8' ) as fp :
				fp.write( self.directXml.toXmlString() )
			fp = None	
			##############################################################
		except :
			if fp is not None :
				fp.close()
			
			self.errorMessages.append( 'File Write Error :' + self.outputFile )
			if not self.forceAction :
				return False
		##############################################################
		return True
	#########################################################################
	def doReload( self ) :
		##############################################################
		if self.printToStd :
			# No Need Reload
			return True
		##############################################################
		if self.skipReloadSameXml and self.isSameXml :
			# No Need Reload
			return True
		##############################################################
		commandList = self.getCommand( 'reloadFirewallD' )
		if len( commandList ) < 1 :
			return True
		##############################################################
		for line in commandList :
			if not self.command.execute( line ) :
				self.errorMessages.append( 'Execution Error :' + line )
				if not self.forceAction :
					return False
				else :
					break
		##############################################################
		return True
	#########################################################################
	def do( self ) :
		##############################################################
		if not self.doLoopBackTcp() :
			return False
		##############################################################
		if not self.doNewNotSynTcp() :
			return False
		##############################################################
		if not self.doNewSynAckTcp() :
			return False
		##############################################################
		if not self.doDropBadTcp() :
			return False
		##############################################################
		if not self.doDropInvalid() :
			return False
		##############################################################
		if not self.doSpecialAllowInputIPv4() :
			return False
		##############################################################
		if not self.doSpecialAllowInputIPv6() :
			return False
		##############################################################
		if not self.doSpecialDenyInputIPv4() :
			return False
		##############################################################
		if not self.doSpecialDenyInputIPv6() :
			return False
		##############################################################
		if not self.doDenyIPv4Countries() :
			return False
		#########################################################################
		if not self.doDenyIPv6Countries() :
			return False
		#########################################################################
		if not self.doSave() :
			return False
		#########################################################################
		if not self.doReload() :
			return False
		#########################################################################
		return True
	#########################################################################
#########################################################################
# end class ActionControl
#########################################################################

#########################################################################
# start class FilePath
#########################################################################
class FilePath :
	#########################################################################
	scriptFullPath = __file__
	scriptDirPath = ''
	scriptRealPath = ''
	scriptRealDirPath = ''
	#########################################################################
	def __init__( self ) :
		self.scriptFullPath = os.path.abspath( __file__ )
		self.scriptDirPath = os.path.dirname( self.scriptFullPath )
		if self.scriptDirPath.endswith('/') :
			self.scriptDirPath = self.scriptDirPath.rstrip('/')

		self.scriptRealPath = os.path.realpath( self.scriptFullPath )
		self.scriptRealDirPath = os.path.dirname( self.scriptRealPath )
		if self.scriptRealDirPath.endswith('/') :
			self.scriptRealDirPath = self.scriptRealDirPath.rstrip('/')
	#########################################################################
#########################################################################
# end class ShellCommand
#########################################################################

#########################################################################
# Start Main
#########################################################################
if __name__ == '__main__':
	
	defaultConfigFile = '/etc/firewalld_ipblock/firewalld_ipblock.conf'
	#########################################################################
	parser = argparse.ArgumentParser( description='FirewallD 用に、指定された国や迷惑パケットをブロックする XML 設定を作成ツールです。' )
	parser.add_argument( '-v', '--version', action='version', version='%(prog)s 1.0')
	parser.add_argument( '-c', '--config', help='設定ファイルを指定する', default='', nargs=1 )
	parser.add_argument( '-o', '--out', help='作成される XML 設定ファイルを指定する', default='', nargs=1 )
	parser.add_argument( '-s', '--screen', help='作成された XML を表示する', action='store_true', default=False )
	parser.add_argument( '-n', '--noreload', help='FirewallD サービスのリロードを行わない', action='store_true', default=False )
	args = parser.parse_args()
	#########################################################################
	if args.config in ( '', [], [''] ) :


		pathAction = FilePath()
		if pathAction.scriptRealDirPath in ( '/usr/lib/firewalld_ipblock' ) :
			# スクリプの軌道場所が /usr/lib/firewalld_ipblock でも /etc を見に行くようにする
			configFile = defaultConfigFile
		else :
			configFile = defaultConfigFile
			if configFile.startswith( '/' ) :
				configFile = configFile[1:]
			configFile = os.path.join( pathAction.scriptRealDirPath, '../../', configFile )

	elif isinstance( args.config, list ) :
		configFile = args.config[0]
	elif isinstance( args.config, str ) :
		configFile = args.config
	else :
		configFile = defaultConfigFile
	#########################################################################
	configFile = os.path.abspath( configFile )

	if not os.path.isfile( configFile ) or not os.access( configFile, os.R_OK ) :
		sys.stderr.write( '設定ファイルが読み込めません : ' +  configFile + '\n' )
		exit( 1 )
		
	#########################################################################
	elif isinstance( args.out, list ) :
		xmlFileTo = args.out[0]
	elif isinstance( args.out, str ) :
		xmlFileTo = args.out
	else :
		xmlFileTo = '/etc/firewalld/direct.xml'
	#########################################################################
	action = ActionControl( configFile = configFile, printToStd = args.screen, outputFile = xmlFileTo )
	ret = action.do()
	#########################################################################
	for line in action.errorMessages :
		sys.stderr.write( line + '\n' )
	#########################################################################
	if ret :
		exit( 0 )
	else :
		exit( 1 )


