#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# ipblock.py version 2.0.0
# Copyright (C) 2014 かんら・から  twitter:@kanrakara 
# 
# ipblock is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# ipblock is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

import urlparse
import urllib
import urllib2
import re
import subprocess
import shlex
import os
import sys
import argparse
import time

#########################################################################
# start class IpCountries
#########################################################################
class IpCountries:
	#########################################################################
	errorMessages = []
	dataSource = []
	countories = []
	ipV4File = None
	ipV6File = None
	curCountry = ''
	ipv4FilePath = ''
	ipv6FilePath = ''
	ipv4DataLength = 0
	ipv6DataLength = 0
	ipv4spltLengthList = []
	#########################################################################
	def __init__( self ) :
		self.reset()

		self.ipv4spltLengthList = []
		for shiftBit in range( 32 ) :
			self.ipv4spltLengthList.append( 1L << shiftBit )
		self.ipv4spltLengthList.reverse()
	#########################################################################
	def __del__( self ) :
		if self.ipV4File :
			self.ipV4File.close()
		if self.ipV6File :
			self.ipV6File.close()
	#########################################################################
	def reset( self ) :
		##############################################################
		if self.ipV4File :
			self.ipV4File.close()
		if self.ipV6File :
			self.ipV6File.close()
		##############################################################
		self.errorMessages = []
		self.dataSource = []
		self.countories = []
		self.ipV4File = None
		self.ipV6File = None
		self.curCountry = ''
		self.ipv4FilePath = ''
		self.ipv6FilePath = ''
		self.ipv4DataLength = 0
		self.ipv6DataLength = 0	
		##############################################################
		
	#########################################################################
	def hasError( self ) :
		return len( self.errorMessages ) > 0
	#########################################################################
	def getErrorMessages( self ) :
		return self.errorMessages
	#########################################################################
	def addDataSource( self, url ):
		##############################################################
		if not isinstance( url, str ) :
			return False
		##############################################################
		added = False
		urlList = re.split( '\r\n|\n', url )
		for urlRow in urlList :
			##############################################################
			urlRow = urlRow.strip()
			urlParsed = urlparse.urlparse( urlRow )
			##############################################################
			if not urlParsed.path :
				continue
			if urlRow in self.dataSource :
				continue
			##############################################################
			self.dataSource.append( urlRow )
			added = True
		##############################################################
		return added
	#########################################################################
	def addCountries( self, country ) :
		##############################################################
		if not isinstance( country, str ) :
			return False
		##############################################################
		countryList = re.split( '\r\n|\n', country )
		for urlRow in countryList :
			##############################################################
			countryRow = urlRow.strip().upper()
			if len( countryRow ) != 2 :
				continue
			##############################################################
			if countryRow in self.countories :
				continue
			##############################################################
			self.countories.append( countryRow )
			added = True
		##############################################################
		return added
	#########################################################################
	def getCountries( self ) :
		return self.countories
	#########################################################################
	def getIPv4DataCount( self ) :
		return self.ipv4DataLength
	#########################################################################
	def getIPv6DataCount( self ) :
		return self.ipv6DataLength
	#########################################################################
	def loadData( self, ipv4Path, ipv6Path, cacheMaxDay = 1 ) :
		##############################################################
		if not isinstance( ipv4Path, str ) :
			return False
		if not isinstance( ipv6Path, str ) :
			return False
		##############################################################
		if os.path.isfile( ipv4Path ) and os.path.isfile( ipv6Path ) :

			ipv4PastSec = time.time() - os.path.getctime( ipv4Path )
			ipv6PastSec = time.time() - os.path.getctime( ipv6Path )
			cacheMaxDSec = cacheMaxDay * 24 * 60 * 60

			if ipv4PastSec < cacheMaxDSec and ipv6PastSec < cacheMaxDSec :
				##############################################################
				self.ipv4FilePath = ipv4Path
				self.ipv6FilePath = ipv6Path
				self.ipv4DataLength = 0
				self.ipv6DataLength = 0
				##############################################################
				with open( self.ipv4FilePath ) as fp :
					for line in fp :
						self.ipv4DataLength += 1
				##############################################################
				with open( self.ipv6FilePath ) as fp :
					for line in fp :
						self.ipv6DataLength += 1
				##############################################################
				return True
		##############################################################
		try :
			##############################################################
			if self.ipV4File :
				self.ipV4File.close()
			if self.ipV6File :
				self.ipV6File.close()
			##############################################################
		except :
			##############################################################
				self.ipV4File = None
				self.ipV6File = None
			##############################################################
		try :
			##############################################################
			self.ipV4File = open( ipv4Path, "w" )
			self.ipV6File = open( ipv6Path, "w" )
			##############################################################
			self.ipv4FilePath = ipv4Path
			self.ipv6FilePath = ipv6Path
			self.ipv4DataLength = 0
			self.ipv6DataLength = 0	
			##############################################################
			for url in self.dataSource :
				##############################################################
				stream = None
				line = 'Open URL : ' + url
				try :
					stream = urllib2.urlopen( url )
				except :
					self.errorMessages.append( "URL Open error : " + url )
					continue
				##############################################################
				for line in stream :
					##############################################################
					# Use Extened Format
					# see : http://www.apnic.net/publications/media-library/documents/resource-guidelines/rir-statistics-exchange-format
					##############################################################
					if len( line ) < 7 :
						continue
					if line[0] == '#' :
						continue
					##############################################################
					line = line.rstrip()
					record = line.rstrip().split( '|' )
					if len( record ) < 7 :
						continue
					##############################################################
					country = record[1]
					if country not in self.countories :
						continue
					##############################################################
					ipType = record[2].lower()
					if ipType not in [ 'ipv4', 'ipv6' ] :
						continue
					##############################################################
					ipString = record[3]
					assignLength = int ( record[4] )
					##############################################################
					if ipType == 'ipv4' :
						##############################################################
						for newIpWithMaskString in self.ipv4StrToWithMask( ipString, assignLength ) :
							##############################################################
							if len( newIpWithMaskString ) < 0 or newIpWithMaskString == '0.0.0.0'  :
								continue
							##############################################################
							self.ipV4File.write( country + '|' + newIpWithMaskString + "\n" )
							self.ipv4DataLength += 1
							##############################################################
						##############################################################
					elif ipType == 'ipv6' :
						maskStr = country + '|' + ipString + "/" + str( assignLength ) + "\n"
						if len( maskStr ) > 0 :
							self.ipV6File.write( maskStr )
							self.ipv6DataLength += 1
						else :
							self.errorMessages.append( "IPv6 Eorror : " + line )
					else :
						continue
				##############################################################
				# line end for
				##############################################################
				stream.close()
			##############################################################
			# dataSource end for
			##############################################################
			self.ipV4File.flush()
			self.ipV6File.flush()
			##############################################################
			self.ipV4File.close()
			self.ipV6File.close()
			##############################################################
			self.ipV4File = None
			self.ipV6File = None
			##############################################################
		except :
			##############################################################
			if self.ipV4File :
				self.ipV4File.close()
			if self.ipV6File :
				self.ipV6File.close()
			##############################################################
			self.ipV4File = None
			self.ipV6File = None
			##############################################################
			self.errorMessages.append( "Data Create Error : " + line )
			##############################################################
			return False
			##############################################################
		return True
	#########################################################################
	def ipv4StrToNum( self, ipv4Str ) :
		##############################################################
		if not isinstance( ipv4Str, str ) :
			return 0L
		##############################################################
		ipv4PatrList = ipv4Str.split('.')
		if len( ipv4PatrList ) != 4 :
			return 0L
		##############################################################
		ip = 0L
		for partStr in ipv4PatrList:
			ip <<= 8
			ip |= int( partStr )
		##############################################################
		return ip
	#########################################################################
	def ipv4NumToStr( self, ipv4Num ) :
		##############################################################
		if not isinstance( ipv4Num, int ) and not isinstance( ipv4Num, long ) :
			return '0.0.0.0'
		##############################################################
		if ipv4Num < 1 :
			return '0.0.0.0'
		##############################################################
		ipv4StrList = []
		num = ipv4Num
		for i in range( 4 ) :
			ipv4StrList = [ str( num % 256 ) ] + ipv4StrList
			num /= 256

		return '.'.join( ipv4StrList )
	#########################################################################
	def ipv4StrToWithMask( self, startIPv4Str, assignLength, ipv4StrWithMaskList = None ) :
		##############################################################
		if ipv4StrWithMaskList is None :
			ipv4StrWithMaskList = []
		##############################################################
		if not isinstance( startIPv4Str, str ) :
			return ipv4StrWithMaskList
		##############################################################
		if not isinstance( assignLength, int ) and not isinstance( assignLength, long ) :
			return ipv4StrWithMaskList
		##############################################################
		if assignLength <= 1 :
			ipv4StrWithMaskList.append( startIPv4Str )
			return ipv4StrWithMaskList
		##############################################################
		if assignLength not in self.ipv4spltLengthList :
			##############################################################
			# need split
			##############################################################
			ipNum = self.ipv4StrToNum( startIPv4Str )
			if ipNum < 1 :
				return ipv4StrWithMaskList
			
			bitHit = 1L
			maskLength = 32
			while not ( ipNum & bitHit ) :
				bitHit <<= 1
				maskLength -= 1

			##############################################################
			if ( 0x01L << ( 32 - maskLength ) ) < assignLength :

				ipv4StrWithMaskList.append( startIPv4Str + '/' + str( maskLength ) )
			
				ipNum += bitHit

				nextAssignLength = assignLength - bitHit
				nextIPv4Str = self.ipv4NumToStr( ipNum )

				return self.ipv4StrToWithMask( nextIPv4Str, nextAssignLength, ipv4StrWithMaskList )
			##############################################################
			for newAssignLength in self.ipv4spltLengthList :
				if newAssignLength <= assignLength :
					break
			##############################################################
			ipv4StrWithMaskList = self.ipv4StrToWithMask( startIPv4Str, newAssignLength, ipv4StrWithMaskList )
			##############################################################
			nextAssignLength = assignLength - newAssignLength
			nextIPv4Str = self.ipv4NumToStr( ipNum + newAssignLength )
			##############################################################
			return self.ipv4StrToWithMask( nextIPv4Str, nextAssignLength, ipv4StrWithMaskList )
			##############################################################
		##############################################################
		rangeMask = assignLength - 1
		ipNum = self.ipv4StrToNum( startIPv4Str )
		##############################################################
		if ipNum & rangeMask :
			##############################################################
			# Overfllow
			##############################################################
			maskLength = 32
			shitedNum = ipNum
			curAssignLength = 0x01
			while not ( shitedNum & 0x01L ) and maskLength > 0 :
				maskLength -= 1
				shitedNum >>= 1
				curAssignLength <<= 1
			
			ipv4StrWithMaskList.append( startIPv4Str + '/' + str( maskLength ) )
			
			nextAssignLength = assignLength - curAssignLength
			nextIPv4Str = self.ipv4NumToStr( ipNum + curAssignLength )

			return self.ipv4StrToWithMask( nextIPv4Str, nextAssignLength, ipv4StrWithMaskList )
			##############################################################
		else :
			##############################################################
			maskLength = 32
			bitCheck = rangeMask
			while bitCheck > 0 :
				maskLength -= 1
				bitCheck >>= 1

			ipv4StrWithMaskList.append( startIPv4Str + '/' + str( maskLength ) )

			return ipv4StrWithMaskList
			##############################################################
	#########################################################################
	def getCountryIP( self, country, ipType = 'ipv4' ) :
		##############################################################
		if not isinstance( country, str ) :
			return False
		if len( country ) != 2 :
			return False
		if not isinstance( ipType, str ) :
			return False
		if len( ipType ) != 4 :
			return False
		##############################################################
		country = country.upper()
		ipType = ipType.lower()
		##############################################################
		if country not in self.countories :
			return False
		if ipType not in [ 'ipv4', 'ipv6' ] :
			return False
		##############################################################
		try :
			##############################################################
			filePtr = None
			if ipType == 'ipv4' :
				if self.ipV4File == None :
					self.ipV4File = open( self.ipv4FilePath, "r" )
				filePtr = self.ipV4File
			##############################################################
			elif ipType == 'ipv6' :
				if self.ipV6File == None :
					self.ipV6File = open( self.ipv6FilePath, "r" )
				filePtr = self.ipV6File
			##############################################################
			else :
				return False
			##############################################################
			if self.curCountry != country :
				filePtr.seek(0)
				self.curCountry = country
			##############################################################
			for line in filePtr :
				line = line.strip()
				##############################################################
				if len( line ) < 1 :
					continue
				record = line.split( '|' )
				if len( record ) < 2 :
					continue
				if record[0] != country :
					continue
				if len( record[1] ) < 1 :
					continue
				##############################################################
				return record[1]
			##############################################################
			return ''
		##############################################################
		except :
			return False
		##############################################################
		return False
	#########################################################################
#########################################################################
# end class IpCountries
#########################################################################

#########################################################################
# start class ShellCommand
#########################################################################
class ShellCommand :
	#########################################################################
	regexpEnv = None
	printToStd = False
	printCommandStd=False
	nullDevice = None
	#########################################################################
	def __init__( self, printToStd = False, printCommandStd = False ) :
		self.regexpEnv = re.compile( '\\$\\{([A-Za-z_][A-Za-z0-9_]*)\\}' )
		self.printToStd = printToStd
		self.printCommandStd = printCommandStd
		if not self.printCommandStd :
			self.nullDevice = open( os.devnull, 'wb' )
	#########################################################################
	def __del__( self ):
		if self.nullDevice :
			self.nullDevice.close()
			self.nullDevice = None
	#########################################################################
	def execute( self, command, replace={} ) :
		##############################################################
		if not isinstance( command, str ) :
			return False
		if not isinstance( replace, dict ) :
			return False
		##############################################################
		command = command.strip()
		if len( command ) < 1 :
			return False

		replaceTarget = self.regexpEnv.findall( command )
		for envTarget in replaceTarget :
			if envTarget in replace :
				command = command.replace( '${' + envTarget + '}', replace[envTarget] )
			elif envTarget in os.environ :
				command = command.replace( '${' + envTarget + '}', os.environ[envTarget] )
			else :
				command = command.replace( '${' + envTarget + '}', '' )
		##############################################################
		avoidError = False
		if command.startswith( '@' ) :
			command = command[1:]
			avoidError = True
		##############################################################
		argList = []
		for arg in shlex.split( command ) :
			##############################################################
			if len( arg ) < 1 :
				continue
			##############################################################
			argList.append( arg )
		##############################################################
		if len( argList ) < 1 :
			return False
		##############################################################
		if self.printToStd :
			regexpEsc = re.compile( '[\\s\\[\\]\\{\\}\"\']+' )
			regexpSingle = re.compile( '(?<!\\\\)\'' )
			printList = []
			for arg in argList :
				if regexpEsc.search( arg ) :
					arg = regexpSingle.sub( "\\'", arg )
					arg = "'" + arg + "'"
				printList.append( arg )
			
			print ' '.join( printList )
			return True
		##############################################################
		retCode = subprocess.call( argList, stdout=self.nullDevice )
		if retCode == 0 or avoidError:
			return True
		else :
			return False
	#########################################################################
#########################################################################
# end class ShellCommand
#########################################################################

#########################################################################
# start class ActionControl
#########################################################################
class ActionControl :
	#########################################################################
	errorMessages = []
	configFile = ''
	config = None
	configDir = ''
	lineSeparatorRegexp = None
	command = None
	forceAction = True
	needReset = True
	dropBadTcp = True
	dropBadInput = True
	denyCountries = None
	printToStd = False
	printCommandStd=False
	cacheMaxDay = 1
	#########################################################################
	def __init__( self, configFile='/usr/local/etc/ipblock/ipblock.conf', printToStd = False, printCommandStd = False ) :
		##############################################################
		self.configFile = configFile
		self.printToStd = printToStd
		self.printCommandStd = printCommandStd
		self.configDir = os.path.dirname( configFile )
		##############################################################
		import ConfigParser
		##############################################################
		self.errorMessages = []
		self.config = ConfigParser.ConfigParser()
		self.config.read( self.configFile )
		##############################################################
		cacheMaxDay = self.getParam ( 'cacheAliveDate' )
		if cacheMaxDay.isdigit() :
			self.cacheMaxDay = int( cacheMaxDay )
		else :
			self.cacheMaxDay = 1
		##############################################################
		self.lineSeparatorRegexp = re.compile( '\r\n|\n' )
		##############################################################
		self.forceAction = self.getFlag ( 'forceAction' )
		self.needReset = self.getFlag ( 'doRest' )
		self.dropBadTcp = self.getFlag ( 'dropBadTcp' )
		self.dropBadInput = self.getFlag ( 'dropBadInput' )
		self.needServiceCheck = not self.getFlag ( 'skipServiceCheck' )
		##############################################################
		self.command = ShellCommand( self.printToStd, self.printCommandStd )
		##############################################################
		self.denyCountries = IpCountries()
		try :
			self.denyCountries.addDataSource( self.config.get( 'dataSource', 'url' ) )
			self.denyCountries.addCountries( self.config.get( 'countries', 'deny' ) )
		except :
			self.denyCountries.reset()
	#########################################################################
	def getFlag( self, option ) :
		##############################################################
		if not isinstance( option, str ) :
			return False
		##############################################################
		try :
			flag = self.config.get( 'actionFlag', option )
			flag = flag.lower()
			if flag == 'false' or flag == 'deny' or flag == '0' :
				return False
			else :
				return True
		except :
			return False
	#########################################################################
	def getCommand( self, option ) :
		##############################################################
		if not isinstance( option, str ) :
			return []
		##############################################################
		try :
			commandLines = self.config.get( 'command', option )
			commandList = []
			for line in self.lineSeparatorRegexp.split( commandLines ) :
				if len( line ) < 1 :
					continue
				else :
					commandList.append( line )
			return commandList
		except :
			return []
	#########################################################################
	def getParam( self, option ) :
		##############################################################
		if not isinstance( option, str ) :
			return ''
		##############################################################
		try :
			flag = self.config.get( 'actionParam', option )
			return flag.strip()
		except :
			return ''
	#########################################################################
	def doReset( self ) :
		##############################################################
		if not self.needReset :
			return True
		##############################################################
		resetCommand = self.getCommand( 'ipv4FirewallRest' )
		resetCommand += self.getCommand( 'ipv6FirewallRest' )
		for line in resetCommand :
			if not self.command.execute( line ) :
				# command error
				self.errorMessages.append( 'Execution Error :' + line )
				if not self.forceAction :
					return False
				else :
					break
		##############################################################
		return True
	#########################################################################
	def doServiceCheck( self ) :
		##############################################################
		if not self.needServiceCheck :
			return True
		##############################################################
		if self.printToStd :
			return True
		##############################################################
		serviceCheckCommand = self.getCommand( 'ipv4ServiceCheck' )
		serviceCheckCommand += self.getCommand( 'ipv6ServiceCheck' )
		for line in serviceCheckCommand :
			if not self.command.execute( line ) :
				# command error
				self.errorMessages.append( 'Execution Error :' + line )
				if not self.forceAction :
					return False
				else :
					break
		##############################################################
		return True
	#########################################################################
	def doDropBadTcp( self ) :
		##############################################################
		if not self.dropBadTcp :
			return True
		##############################################################
		drpBadTcpCommand = self.getCommand( 'ipv4DropBadTcp' )
		drpBadTcpCommand += self.getCommand( 'ipv6DropBadTcp' )
		for line in drpBadTcpCommand :
			if not self.command.execute( line ) :
				self.errorMessages.append( 'Execution Error :' + line )
				if not self.forceAction :
					return False
				else :
					break
		##############################################################
		return True
	#########################################################################
	def doDropBadInput( self ) :
		##############################################################
		if not self.dropBadInput :
			return True
		##############################################################
		drpBadInputCommand = self.getCommand( 'ipv4DropBadInput' )
		drpBadInputCommand += self.getCommand( 'ipv6DropBadInput' )
		for line in drpBadInputCommand :
			if not self.command.execute( line ) :
				self.errorMessages.append( 'Execution Error :' + line )
				if not self.forceAction :
					return False
				else :
					break
		##############################################################
		return True
	#########################################################################
	def doSpecialAction( self, dataFilePath, commandName ) :
		try :
			filePathList = self.config.get( 'files', dataFilePath )
			for filePath in self.lineSeparatorRegexp.split( filePathList ) :
				filePath = filePath.strip()
				filePath = os.path.join( self.configDir, filePath )
				if not os.path.exists( filePath ) :
					self.errorMessages.append( 'doSpecialAction Load Error : ' + filePath )
					if not self.forceAction :
						return False
					else :
						continue

				command = self.getCommand( commandName )
				if len( command ) < 1 :
					return True
			
				with open( filePath, "r" ) as fp :
				##############################################################
					for address in fp :
						##############################################################
						address = address.strip()
						if len( address ) < 1 :
							continue
						elif address[0] == '#' :
							continue
						##############################################################
						replace = { 'TargetAddress' : address }
						##############################################################
						for line in command :
							if not self.command.execute( line, replace ) :
								self.errorMessages.append( 'Execution Error :' + line )
								if not self.forceAction :
									return False
								else :
									break
						##############################################################
			##############################################################
			return True
			##############################################################
		except :
			return False
	#########################################################################
	def doSpecialAllowInputIPv4( self ) :
		return self.doSpecialAction( 'dataAllowFileIPv4', 'ipv4SpecialIpAccept' )
	#########################################################################
	def doSpecialAllowInputIPv6( self ) :
		return self.doSpecialAction( 'dataAllowFileIPv6', 'ipv6SpecialIpAccept' )
	#########################################################################
	def doSpecialDenyInputIPv4( self ) :
		return self.doSpecialAction( 'dataDenyFileIPv4', 'ipv4SpecialIpDrop' )
	#########################################################################
	def doSpecialDenyInputIPv6( self ) :
		return self.doSpecialAction( 'dataDenyFileIPv6', 'ipv6SpecialIpDrop' )
	#########################################################################
	def loadDenyCountries( self, forceReload = False ) :
		##############################################################
		if self.denyCountries is None :
			self.denyCountries = IpCountries()
		##############################################################
		if forceReload is False :
			if self.denyCountries.getIPv4DataCount() > 0 :
				return True
			if self.denyCountries.getIPv6DataCount() > 0 :
				return True
		##############################################################
		try :
			#########################################################################
			self.denyCountries.reset()
			#########################################################################
			self.denyCountries.addDataSource( self.config.get( 'dataSource', 'url' ) )
			self.denyCountries.addCountries( self.config.get( 'countries', 'deny' ) )
			#########################################################################
			dataFileIPv4 = self.config.get( 'files', 'dataFileIPv4' )
			dataFileIPv6 = self.config.get( 'files', 'dataFileIPv6' )
			#########################################################################
			results = self.denyCountries.loadData( dataFileIPv4, dataFileIPv6, self.cacheMaxDay )
			if not results :
				return False
			elif not self.forceAction and self.denyCountries.hasError() :
				return False
			#########################################################################
			return True
			#########################################################################
		except :
			return False
	#########################################################################
	def doDenyCountryIP( self, startCommand, denyCommand, endCommand, country, ipType ) :
		##############################################################
		if not isinstance( startCommand, list ) :
			return False
		##############################################################
		if not isinstance( denyCommand, list ) :
			return False
		##############################################################
		if not isinstance( endCommand, list ) :
			return False
		##############################################################
		if not isinstance( country, str ) :
			return False
		##############################################################
		if not isinstance( ipType, str ) :
			return False
		ipType = ipType.lower()
		if ipType not in [ 'ipv4', 'ipv6' ] :
			return False
		##############################################################
		if len( denyCommand ) > 0 :
			##############################################################
			if not self.loadDenyCountries() :
				##############################################################
				self.errorMessages.append( 'loadDenyCountries Error ' )
				if self.denyCountries is not None :
					for line in self.denyCountries.getErrorMessages() :
						self.errorMessages.append( 'loadDenyCountries Error : ' + line )
				if not self.forceAction :
					return False
				##############################################################
			##############################################################
			elif country not in self.denyCountries.getCountries() :
				return False
		##############################################################
		for line in startCommand :
			##############################################################
			replace = { 'TargetCountry' : country }
			##############################################################
			if not self.command.execute( line, replace ) :
				self.errorMessages.append( 'Execution Error :' + line )
				if not self.forceAction :
					return False
		##############################################################
		while self.denyCountries is not None :
			ip = self.denyCountries.getCountryIP( country, ipType )
			##############################################################
			if ip is False :
				# Error
				self.errorMessages.append( 'getCountryIP Error :' + country + ':' + ipType )
				if not self.forceAction :
					return False
				else :
					continue
			elif len( ip ) < 1 :
				# End of Data
				break
			##############################################################
			replace = { 'TargetCountry' : country, 'TargetAddress' : ip }
			##############################################################
			for line in denyCommand :
				if not self.command.execute( line, replace ) :
					self.errorMessages.append( 'Execution Error :' + line )
					if not self.forceAction :
						return False
					else :
						break
			##############################################################
		##############################################################
		for line in endCommand :
			##############################################################
			replace = { 'TargetCountry' : country }
			##############################################################
			if not self.command.execute( line, replace ) :
				self.errorMessages.append( 'Execution Error :' + line )
				if not self.forceAction :
					return False
		##############################################################
		return True
	#########################################################################
	def doDenyIPv4Countries( self ) :
		##############################################################
		if self.denyCountries is None :
			return False
		##############################################################
		dropIPv4Start = self.getCommand( 'ipv4DorpTargetCountryStart' )
		dropIPv4Command = self.getCommand( 'ipv4DorpTargetCountry' )
		dropIPv4End = self.getCommand( 'ipv4DorpTargetCountryEnd' )
		##############################################################
		for country in self.denyCountries.getCountries() :
			if not self.doDenyCountryIP( dropIPv4Start, dropIPv4Command, dropIPv4End, country, 'ipv4' ) :
				if not self.forceAction :
					return False
				else :
					continue
		##############################################################
		return True
	#########################################################################
	def doDenyIPv6Countries( self ) :
		##############################################################
		if self.denyCountries is None :
			return False
		##############################################################
		dropIPv6Start = self.getCommand( 'ipv6DorpTargetCountryStart' )
		dropIPv6Command = self.getCommand( 'ipv6DorpTargetCountry' )
		dropIPv6End = self.getCommand( 'ipv6DorpTargetCountryEnd' )
		##############################################################
		for country in self.denyCountries.getCountries() :
			if not self.doDenyCountryIP( dropIPv6Start, dropIPv6Command, dropIPv6End, country, 'ipv6' ) :
				if not self.forceAction :
					return False
				else :
					continue
		##############################################################
		return True
	#########################################################################
	def doSave( self ) :
		##############################################################
		if self.printToStd :
			return True
		##############################################################
		oSaveCommand = self.getCommand( 'ipv4Save' )
		oSaveCommand += self.getCommand( 'ipv6Save' )
		for line in oSaveCommand :
			if not self.command.execute( line ) :
				self.errorMessages.append( 'Execution Error :' + line )
				if not self.forceAction :
					return False
				else :
					break
		##############################################################
		return True
	#########################################################################
	def do( self ) :
		##############################################################
		if not self.doServiceCheck() :
			return False
		##############################################################
		if not self.doReset() :
			return False
		##############################################################
		if not self.doDropBadTcp() :
			return False
		##############################################################
		if not self.doDropBadInput() :
			return False
		##############################################################
		if not self.doSpecialAllowInputIPv4() :
			return False
		##############################################################
		if not self.doSpecialAllowInputIPv6() :
			return False
		##############################################################
		if not self.doSpecialDenyInputIPv4() :
			return False
		##############################################################
		if not self.doSpecialDenyInputIPv6() :
			return False
		##############################################################
		if not self.doDenyIPv4Countries() :
			return False
		#########################################################################
		if not self.doDenyIPv6Countries() :
			return False
		#########################################################################
		if not self.doSave() :
			return False
		#########################################################################
		return True
	#########################################################################
#########################################################################
# end class ActionControl
#########################################################################

#########################################################################
# start class FilePath
#########################################################################
class FilePath :
	#########################################################################
	scriptFullPath = __file__
	scriptDirPath = ''
	scriptRealPath = ''
	scriptRealDirPath = ''
	#########################################################################
	def __init__( self ) :
		self.scriptFullPath = os.path.abspath( __file__ )
		self.scriptDirPath = os.path.dirname( self.scriptFullPath )
		if self.scriptDirPath.endswith('/') :
			self.scriptDirPath = self.scriptDirPath.rstrip('/')

		self.scriptRealPath = os.path.realpath( self.scriptFullPath )
		self.scriptRealDirPath = os.path.dirname( self.scriptRealPath )
		if self.scriptRealDirPath.endswith('/') :
			self.scriptRealDirPath = self.scriptRealDirPath.rstrip('/')
	#########################################################################
#########################################################################
# end class ShellCommand
#########################################################################

#########################################################################
# Start Main
#########################################################################
if __name__ == '__main__':

	if os.geteuid() != 0 :
		sys.stderr.write( 'このツールを利用するには管理者権限が必要です。\n' )
		exit( 1 )

	defaultConfigFile='/etc/ipblock/ipblock.conf'

	#########################################################################
	parser = argparse.ArgumentParser( description='指定された国や迷惑パケットをブロックする設定を行うツールです。' )
	parser.add_argument( '-v', '--version', action='version', version='%(prog)s 2.0')
	parser.add_argument( '-c', '--config', help='設定ファイルを指定する', default='', nargs=1 )
	parser.add_argument( '-s', '--screen', help='コマンドを実行せず表示する', action='store_true', default=False )
	parser.add_argument( '-n', '--nocheck', help='サービスのチェックを行わない', action='store_true', default=False )
	parser.add_argument( '-l', '--loud', help='実行したコマンドの出力を表示する', action='store_true', default=False )
	args = parser.parse_args()
	#########################################################################
	if args.config in ( '', [], [''] ) :

		configFile = defaultConfigFile
		if configFile.startswith( '/' ) :
			configFile = configFile[1:]

		pathAction = FilePath()
		if pathAction.scriptRealDirPath in ( '/usr/lib/ipblock' ) :
			# スクリプの軌道場所が /usr/lib/ipblock でも /etc を見に行くようにする
			configFile = defaultConfigFile
		else :
			configFile = os.path.join( pathAction.scriptRealDirPath, '../../', configFile )

	elif isinstance( args.config, list ) :
		configFile = args.config[0]
	elif isinstance( args.config, str ) :
		configFile = args.config
	else :
		configFile = defaultConfigFile
	#########################################################################

	configFile = os.path.abspath( configFile )

	if not os.path.isfile( configFile ) or not os.access( configFile, os.R_OK ) :
		sys.stderr.write( '設定ファイルが読み込めません : ' +  configFile + '\n' )
		exit( 1 )
		
	#########################################################################
	action = ActionControl( configFile=configFile, printToStd=args.screen, printCommandStd=args.loud )
	ret = action.do()
	#########################################################################
	for line in action.errorMessages :
		sys.stderr.write( line + '\n' )
	#########################################################################
	if ret :
		exit( 0 )
	else :
		exit( 1 )
