#!/usr/bin/env python3
# -*- coding: utf-8 -*-
########################################################################
# version 1.0.0
#
# gimp3-tensha.py
# Copyright (C) 2025 かんら・から https://www.magic-object.com/
#
# gimp3-tensha.py is Python script plugin for GIMP 3.0
#
# gimp3-tensha.py is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# gimp3-tensha.py is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# GPLv3 ライセンス
# かんら・から https://www.magic-object.com/fedora17withextendpythonfu1/
# バグレポート・アイデアなどはコメントでお願いします。
#
# ダウンロード
# https://www.magic-object.com/
#
# このスクリプトを使用して発生した問題に関し、作成者は如何なる保証などを行う事はありません。
# 自己責任でのみの使用を許可します。
########################################################################
import sys

import gi

gi.require_version('Gimp', '3.0')
from gi.repository import Gimp

gi.require_version('Gimp', '3.0')
from gi.repository.Gimp import MessageHandlerType

gi.require_version('Gimp', '3.0')
from gi.repository.Gimp import MergeType

gi.require_version('GimpUi', '3.0')
from gi.repository import GimpUi

from gi.repository import GLib


########################################################################
# 線画の転写プラグイン
#
class Gimp3TenshaPlugin(Gimp.PlugIn):
    def do_query_procedures(self):
        return ["tensha"]

    def do_set_i18n(self, name):
        return False

    ########################################################################
    # プラグインの登録
    #
    def do_create_procedure(self, name):
        procedure = Gimp.ImageProcedure.new(self, name, Gimp.PDBProcType.PLUGIN, self.tensha, None)

        procedure.set_image_types("RGBA")

        procedure.set_menu_label("線画の転写")
        procedure.add_menu_path('<Image>/Layer/レイヤー操作/')

        procedure.set_documentation("線画の転写", "Python 3 plug-in for GIMP 3.0", name)
        procedure.set_attribution("かんら・から", "Magic Object", "2025")

        return procedure

    ########################################################################
    # メッセージ機能
    #
    def message(self, message:str):
        Gimp.message_set_handler(Gimp.MessageHandlerType.MESSAGE_BOX)
        Gimp.message(message)
        # MESSAGE_BOX CONSOLE ERROR_CONSOLE
        # gimp.message( message )
        return

    def tensha(self, procedure, run_mode, image:Gimp.Image, drawables, config, run_data):
        if drawables is None:
            self.message("レイヤーが選択されていません。")
            return procedure.new_return_values(Gimp.PDBStatusType.EXECUTION_ERROR, GLib.Error())
        elif len(drawables) > 1:
            self.message("複数レイヤーは指定できません。")
            return procedure.new_return_values(Gimp.PDBStatusType.EXECUTION_ERROR, GLib.Error())
        elif drawables[0].is_text_layer():
            self.message("テキストレイヤーは指定できません。")
            return procedure.new_return_values(Gimp.PDBStatusType.EXECUTION_ERROR, GLib.Error())
        elif drawables[0].is_group():
            self.message("グループレイヤーは指定できません。")
            return procedure.new_return_values(Gimp.PDBStatusType.EXECUTION_ERROR, GLib.Error())
        elif not drawables[0].has_alpha():
            self.message("アルファチャンネルがないレイヤーは指定できません。")
            return procedure.new_return_values(Gimp.PDBStatusType.EXECUTION_ERROR, GLib.Error())
        elif not drawables[0].is_layer():
            self.message("レイヤー以外は指定できません。")
            return procedure.new_return_values(Gimp.PDBStatusType.EXECUTION_ERROR, GLib.Error())

        layer = drawables[0]
        parent = layer.get_parent()
        item_id = layer.get_id()

        under_layer = None
        hit = False
        position = 0
        for child in parent.get_children():
            if hit:
                under_layer = child
                break
            elif item_id == child.get_id():
                hit = True
                continue
            else:
                position += 1

        if under_layer is None:
            self.message("末尾のレイヤーは指定できません。")
            return procedure.new_return_values(Gimp.PDBStatusType.EXECUTION_ERROR, GLib.Error())
        ##############################################################
        # 転写元のレイヤーを作成しておく
        # データなしの複製
        new_layer = Gimp.Layer.new(
            image,
            layer.get_name(),
            layer.get_width(),
            layer.get_height(),
            Gimp.ImageType.RGBA_IMAGE,
            layer.get_opacity(),
            layer.get_mode()
        )
        # レイヤーマスクを設定
        if layer.get_mask() is not None:
            new_layer.add_mask(layer.get_mask().copy())
        ##############################################################
        # 転写先のレイヤーマスクを複製
        under_layer_mask = None
        if under_layer.get_mask() is not None:
            under_layer_mask = under_layer.get_mask().copy()
        ##############################################################
        # 転写先のレイヤーモードを保持
        under_layer_mode = under_layer.get_mode()
        ##############################################################
        # 下のレイヤーと統合
        results_layer = image.merge_down(layer, Gimp.MergeType.EXPAND_AS_NECESSARY)
        # レイヤーマスクを設定
        if under_layer_mask is not None:
            results_layer.add_mask(under_layer_mask)
        # レイヤーモードを設定
        results_layer.set_mode(under_layer_mode)
        ##############################################################
        # 転写元のレイヤーを挿入
        image.insert_layer(new_layer, parent, position)
        image.select_item(Gimp.ChannelOps.REPLACE, new_layer)
        ##############################################################
        #self.message("転写しました。")
        # do what you want to do, then, in case of success, return:
        return procedure.new_return_values(Gimp.PDBStatusType.SUCCESS, GLib.Error())


Gimp.main(Gimp3TenshaPlugin.__gtype__, sys.argv)
