#!/usr/bin/env python3
# -*- coding: utf-8 -*-
########################################################################
# version 1.0.0
#
# gimp3-group-mask.py
# Copyright (C) 2025 かんら・から https://www.magic-object.com/
#
# gimp3-group-mask.py is Python script plugin for GIMP 3.0
#
# gimp3-group-mask.py is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# gimp3-group-mask.py is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# GPLv3 ライセンス
# かんら・から https://www.magic-object.com/
# バグレポート・アイデアなどはコメントでお願いします。
#
# ダウンロード
# https://www.magic-object.com/gimp-python-fu-%e6%8b%a1%e5%bc%b5%e2%91%a3/
#
# このスクリプトを使用して発生した問題に関し、作成者は如何なる保証などを行う事はありません。
# 自己責任でのみの使用を許可します。
########################################################################
import sys

import gi

gi.require_version('Gimp', '3.0')
from gi.repository import Gimp

gi.require_version('Gimp', '3.0')
from gi.repository.Gimp import MessageHandlerType

gi.require_version('Gimp', '3.0')
from gi.repository.Gimp import MergeType

gi.require_version('GimpUi', '3.0')
from gi.repository import GimpUi

from gi.repository import GLib


########################################################################
# 線画の転写プラグイン
#
class Gimp3GroupMaskPlugin(Gimp.PlugIn):
    def do_query_procedures(self):
        return ["group-mask"]

    def do_set_i18n(self, name):
        return False

    ########################################################################
    # プラグインの登録
    #
    def do_create_procedure(self, name):
        procedure = Gimp.ImageProcedure.new(self, name, Gimp.PDBProcType.PLUGIN, self.group_mask_proc, None)

        procedure.set_image_types("RGBA")

        procedure.set_menu_label("グループマスクの作成")
        procedure.add_menu_path('<Image>/Layer/レイヤー操作/')

        procedure.set_documentation("グループレイヤー最下部のペイントレイヤーのアルファマスクから、グループマスクの作成を行います。", "Python 3 plug-in for GIMP 3.0", name)
        procedure.set_attribution("かんら・から", "Magic Object", "2025")

        return procedure

    ########################################################################
    # メッセージ機能
    #
    def message(self, message:str):
        Gimp.message_set_handler(Gimp.MessageHandlerType.MESSAGE_BOX)
        Gimp.message(message)
        # MESSAGE_BOX CONSOLE ERROR_CONSOLE
        # gimp.message( message )
        return

    def group_mask(self, procedure, run_mode, image:Gimp.Image, drawables, config, run_data):
        if drawables is None:
            self.message("レイヤーまたはグループが選択されていません。")
            return False
        elif len(drawables) < 1:
            self.message("レイヤーまたはグループが選択されていません。")
            return False
        elif drawables[0].is_text_layer():
            self.message("テキストレイヤーは指定できません。")
            return False
        elif drawables[0].is_layer_mask():
            self.message("レイヤーマスクは指定できません。")
            return False
        elif drawables[0].is_layer() and not drawables[0].is_group() and not drawables[0].has_alpha():
            self.message("アルファチャンネルがないレイヤーは指定できません。")
            return False
        elif drawables[0].is_layer() and not drawables[0].is_group() and drawables[0].get_parent() is None:
            self.message("グループ内に存在しないレイヤーは指定できません。")
            return False
        elif drawables[0].is_layer() and not drawables[0].is_group():
            # グループ内の通常レイヤーの場合はグループを渡す
            return self.group_mask(procedure, run_mode, image, [drawables[0].get_parent()], config, run_data)
        elif not drawables[0].is_group() and drawables[0].get_parent() is None:
            self.message("グループ以外は指定できません。")
            return False
        elif len(drawables) > 1:
            self.message("複数はは指定できません。")
            return False

        name = drawables[0].get_name()
        self.message(f"{name}処理中です。")

        group = drawables[0]
        if group.get_mask() is not None:
            # グループにレイヤーマスクが既に存在する
            return True

        children = group.get_children()
        count = len(children)
        if count < 1:
            # 空のグループレイヤー
            return True

        last_layer = children[count-1]

        is_all_group = True
        for child in children:
            if child.is_group():
                # グループが入れ子になっているので、そのグループにもマスクを作成しておく
                self.group_mask(procedure, run_mode, image, [child], config, run_data)
            else:
                is_all_group = False

        if is_all_group:
            return True

        if not last_layer.has_alpha():
            self.message("アルファチャンネルがないレイヤーグループは指定できません。")
            return False
        elif last_layer.is_text_layer():
            self.message("テキストレイヤーは指定できません。")
            return False

        group_offset_OK, group_offset_x, group_offset_y = group.get_offsets()
        if not group_offset_OK:
            self.message("グループオフセットの取得に失敗しました。")
            return False

        last_layer_offset_OK, last_layer_offset_x, last_layer_offset_y = last_layer.get_offsets()
        if not last_layer_offset_OK:
            self.message("レイヤーオフセットの取得に失敗しました。")
            return False

        # グループのサイズの取得
        group_width = group.get_width()
        group_height = group.get_height()

        # 末尾レイヤーのサイズの取得
        last_layer_width = last_layer.get_width()
        last_layer_height = last_layer.get_height()

        if group_width > last_layer_width or group_height > last_layer_height or group_offset_x > last_layer_offset_x or group_offset_y > last_layer_offset_y:
            # 末尾レイヤーのリサイズ
            last_layer.resize(group_width, group_height, last_layer_offset_x - group_offset_x, last_layer_offset_y - group_offset_y)

        if last_layer.get_mask() is not None:
            group.add_mask(last_layer.get_mask().copy())
            return True
        else:
            group.add_mask(last_layer.create_mask(Gimp.AddMaskType.ALPHA))
            last_layer.set_apply_mask(True)
            return True

    def group_mask_proc(self, procedure, run_mode, image:Gimp.Image, drawables, config, run_data):
        if self.group_mask(procedure, run_mode, image, drawables, config, run_data):
            self.message("終了しました。")
            return procedure.new_return_values(Gimp.PDBStatusType.SUCCESS, GLib.Error())
        else:
            return procedure.new_return_values(Gimp.PDBStatusType.EXECUTION_ERROR, GLib.Error())


Gimp.main(Gimp3GroupMaskPlugin.__gtype__, sys.argv)
